#!/bin/sh
# Statistical analysis of GRASS raster maps
# written by Markus Neteler 1998-2002
#            neteler@itc.it
#

#%Module
#%  description: This module calculates univariate statistics of a raster map
#%End
#%flag
#%  key: e
#%  description: calculate extended statistics (median and quartiles)
#%END
#%option
#% key: cover
#% type: string
#% gisprompt: old,cell,raster
#% description: raster input map
#% required : yes
#%end
#%option
#% key: base
#% type: string
#% gisprompt: old,cell,raster
#% description: raster input map (optional masking map containing areas)
#% required : no
#%end

if [ "$1" != "@ARGS_PARSED@" ] ; then
  exec $GISBASE/etc/bin/cmd/g.parser "$0" "$@"
fi

if test "$GISBASE" = ""; then
 echo "You must be in GRASS GIS to run this program." >&2
 exit 1
fi   
     
eval `g.gisenv`
: ${GISBASE?} ${GISDBASE?} ${LOCATION_NAME?} ${MAPSET?}
LOCATION=$GISDBASE/$LOCATION_NAME/$MAPSET

TMP=/tmp/r.univar.tmp
cleanup()
{
 \rm -f $TMP $TMP.sort
}

# what to do in case of user break:
exitprocedure()
{
 echo "User break!"
 cleanup
 exit 1
}
# shell check for user break (signal list: trap -l)
trap "exitprocedure" 2 3 15

gotitBASE=0
gotitCOVER=0

cover="$GIS_OPT_cover"
base="$GIS_OPT_base"
if [ "$GIS_OPT_base" != "(null)" ] ; then
  gotitBASE=1
  eval `g.findfile element=cell file=$base`
  base="${fullname}"
  BASE="${name}"
  if [ "$base" = "" ] ; then
     echo "ERROR: base raster map [`echo $i | awk -F '=' '{print $2}'`] does not exist."
     exit 1
  fi
fi

gotitCOVER=1
eval `g.findfile element=cell file=$cover`
cover="${fullname}"
COVER="${name}"
if [ "$cover" = "" ] ; then
  echo "ERROR: cover raster map [`echo $i | awk -F '=' '{print $2}'`] does not exist."
  exit 1
fi

#also run if only one map specified, take it as cover map:
if [ $# -eq 1 -a $gotitBASE -eq 0 -a $gotitCOVER -eq 0 ] ; then
       cover=$1
       gotitCOVER=1
       eval `g.findfile element=cell file=$cover`
       cover="${fullname}"
       COVER="${name}"
       if [ "$cover" = "" ] ; then
          echo "ERROR: cover raster map [$1] does not exist."
          exit 1
       fi
fi

#query base map only when in interactive mode:
if [ $gotitCOVER -eq 0 ] ; then
 while test $gotitBASE -eq 0 
  do
   echo ""
   g.ask type=old element=cell desc=raster prompt="Enter base raster map (or return to skip)" unixfile=/tmp/$$
   eval `cat /tmp/$$`
   rm -f /tmp/$$
   if [ ! "$file" ]
   then
   # we don't exit, because we want to skip this map
     gotitBASE=1
   fi
   base="${fullname}"
   BASE="${name}"
   echo "$base"
   gotitBASE=1
  done
fi

# in case of skipped base map, reset gotitBASE:
if [ $gotitBASE -eq 1 -a ! "$base" ]
then
 gotitBASE=0
fi

while test $gotitCOVER -eq 0 
do
 echo ""
 g.ask type=old element=cell desc=raster prompt="Enter cover raster map" unixfile=/tmp/$$
 eval `cat /tmp/$$`
 rm -f /tmp/$$
 if [ ! "$file" ]
 then
    exit 0
 fi
 cover="${fullname}"
 COVER="${name}"
 echo "$cover"
 gotitCOVER=1
done

echo ""
echo "This module calculates univariate statistics..."

if [ $gotitBASE -eq 1 ]
then
   echo "Calculation for images $cover masked through $base..."
   echo "(ignoring NULL cells)"
   echo "Reading raster maps..."
   r.stats -1n input=$base,$cover |grep '1 ' |cut -d' ' -f2 > $TMP
else
   echo "Calculation for full image $cover..."
   echo "(no base mask map set, ignoring NULL cells)"
   echo "Reading raster map..."
   r.stats -1n input=$cover > $TMP
fi

#check if map contains only NULL's in current region
LINES=`wc -l $TMP | awk '{print $1}'`
if [ "$LINES" = 0 ]
then
 echo ""
 echo "ERROR: Map $COVER contains only NULL data in current region."
 cleanup
 exit 1
fi

# calculate statistics
echo "Calculating statistics..."
cat $TMP | awk 'BEGIN {sum = 0.0 ; sum2 = 0.0} 
NR == 1{min = $1 ; max = $1}
       {sum += $1 ; sum2 += $1 * $1 ; N++}
       {
        if ($1 > max) {max = $1}
        if ($1 < min) {min = $1}
       }
END{
print ""
print "Number of cells (excluding NULL cells):",N
print "Minimum:",min
print "Maximum:",max
print "Range:",max-min
print "Arithmetic mean:",sum/N
print "Variance:",(sum2 - sum*sum/N)/N
print "Standard deviation:",sqrt((sum2 - sum*sum/N)/N)
print "Variation coefficient:",100*(sqrt((sum2 - sum*sum/N)/N))/(sqrt(sum*sum)/N),"%"
}'

if [ $GIS_FLAG_e -eq 1 ] ; then
 #preparations:
 cat $TMP | sort -n > $TMP.sort
 NUMBER=`cat $TMP.sort | wc -l | awk '{print $1}'`
 ODDEVEN=`echo $NUMBER | awk '{print $1%2}'`

 # 0.25 quartile
 QUARTILE=0.25
 QPOS=`echo $NUMBER $QUARTILE | awk '{printf "%d", $1 * $2 + 0.5}'`
 QELEMENT=`head -$QPOS $TMP.sort | tail -1`
 echo "1st Quartile: $QELEMENT"

 #Calculate median
 if [ $ODDEVEN -eq 0 ]
 then
  # even
  EVENMEDIANNUMBER=`expr $NUMBER / 2`
  EVENMEDIANNUMBERPLUSONE=`expr $EVENMEDIANNUMBER + 1`
  # select two numbers
  SELECTEDNUMBERS=`cat $TMP.sort | head -$EVENMEDIANNUMBERPLUSONE | tail -2`
  RESULTEVENMEDIAN=`echo $SELECTEDNUMBERS | awk '{printf "%f", ($1 + $2)/2.0}'`
  echo "Median (even N): $RESULTEVENMEDIAN"
 else
  # odd
  ODDMEDIANNUMBER=`echo $NUMBER | awk '{printf "%d", int($1/2+.5)}'`
  RESULTODDMEDIAN=`cat $TMP.sort | head -$ODDMEDIANNUMBER | tail -1 | awk '{printf "%f", $1}'`
  echo "Median (odd N): $RESULTODDMEDIAN"
 fi

 # 0.75 quartile
 QUARTILE=0.75
 QPOS=`echo $NUMBER $QUARTILE | awk '{printf "%d", $1 * $2 + 0.5}'`
 QELEMENT=`head -$QPOS $TMP.sort | tail -1`
 echo "3rd Quartile: $QELEMENT"

 rm -f $TMP.sort
fi

cleanup
