## DESCRIPTION

The module *r3.out.netcdf* exports a 3D raster map as netCDF file. Maps
are valid 3D raster maps in the current mapset search path. The *output*
parameter is the name of a netCDF file that will be written in the
current working directory. The module is sensitive to mapset region
settings (set with *g.region*) and the vertical unit settings of the 3D
raster map. The vertical unit can be set with
[r3.support](r3.support.md).

## NOTES

The resulting netCDF file will have time as third dimension in case a
temporal vertical unit is specified for the input map, otherwise the
third dimension is assumed as spatial. If the 3D raster map has a time
stamp with absolute time, the start time will be used as starting point
for the netCDF time series. If the absolute time stamp is missing the
date "Jan. 01. 1900 00:00:00" will be used as default. In case of
relative time stamp no start time is set in the netCDF file. The number
of depths of the 3D raster map are the number of time slices in the
netCDF file. You can use *t.rast.to.rast3* to convert a space time
raster dataset into a 3D raster map representing a space time voxel
cube, then export the raster time series as netCDF file.

Spatial coordinates are exported as cell centered coordinates. The
projection can be optionally stored in the metadata as *crs* attributes
. The netCDF projection metadata storage follows the [spatial_ref
GDAL/netCDF suggestion](https://gdal.org/en/stable/drivers/raster/netcdf.html)
and the [netCDF CF 1.6 convention](https://cfconventions.org/cf-conventions/v1.6.0/cf-conventions.html#coordinate-system)
using WKT projection information. Additional a PROJ string is stored in
the *crs* attribute section. The export of projection parameters is
suppressed when the XY-projection is set.

The range of the 3D raster map is set in the netCDF output file.
Optionally a the netCDF missing value and \_FillValue can be set using
the *null* option. As default the netCDF floating point \_FillValues are
used to fill empty cell.

The netCDF library must be installed on the system and activated at
configuration time to compile this module.

## EXAMPLES

```sh
g.region s=-90 n=90 w=-180 e=180 b=0 t=5 res=10 res3=10 tbres=1 -p3

#####################################################################
# We create a simple volume map with floating point values
#####################################################################

r3.mapcalc --o expr="volume_float = float(col() + row() + depth())"
r3.info volume_float
 +----------------------------------------------------------------------------+
 | Layer:    volume_float                   Date: Thu Jun 14 08:40:56 2012    |
 | Mapset:   PERMANENT                      Login of Creator: soeren          |
 | Project:  TestLL                                                           |
 | DataBase: /1/soeren/grassdata                                              |
 | Title:    volume_float                                                     |
 | Units:    none                                                             |
 | Vertical unit: units                                                       |
 | Timestamp: none                                                            |
 |----------------------------------------------------------------------------|
 |                                                                            |
 |   Type of Map:  3d cell              Number of Categories: 0               |
 |   Data Type:    FCELL                                                      |
 |   Rows:         18                                                         |
 |   Columns:      36                                                         |
 |   Depths:       5                                                          |
 |   Total Cells:  3240                                                       |
 |   Total size:           10114 Bytes                                        |
 |   Number of tiles:      1                                                  |
 |   Mean tile size:       10114 Bytes                                        |
 |   Tile size in memory:  12960 Bytes                                        |
 |   Number of tiles in x, y and  z:   1, 1, 1                                |
 |   Dimension of a tile in x, y, z:   36, 18, 5                              |
 |                                                                            |
 |        Projection: Latitude-Longitude (zone 0)                             |
 |            N:        90N    S:        90S   Res:    10                     |
 |            E:       180E    W:       180W   Res:    10                     |
 |            T:          5    B:          0   Res:     1                     |
 |   Range of data:   min =          3 max =         59                       |
 |                                                                            |
 |   Data Source:                                                             |
 |                                                                            |
 |                                                                            |
 |                                                                            |
 |   Data Description:                                                        |
 |    generated by r3.mapcalc                                                 |
 |                                                                            |
 |   Comments:                                                                |
 |    r3.mapcalc expression="volume_float = float(col() + row() + depth())"   |
 |                                                                            |
 +----------------------------------------------------------------------------+

#####################################################################
# We use the netCDF ncdump tool to have a look at the header
# and coordinates of the exported netCDF file
#####################################################################

r3.out.netcdf --o input=volume_float output=test_float.nc
ncdump -c test_float.nc

netcdf test_float {
dimensions:
        longitude = 36 ;
        latitude = 18 ;
        z = 5 ;
variables:
        float longitude(longitude) ;
                longitude:units = "degrees_east" ;
                longitude:long_name = "Longitude values" ;
                longitude:standard_name = "longitude" ;
                longitude:axis = "X" ;
        float latitude(latitude) ;
                latitude:units = "degrees_north" ;
                latitude:long_name = "Latitude values" ;
                latitude:standard_name = "latitude" ;
                latitude:axis = "Y" ;
        float z(z) ;
                z:units = "meter" ;
                z:long_name = "z coordinate of projection" ;
                z:standard_name = "projection_z_coordinate" ;
                z:positive = "up" ;
                z:axis = "Z" ;
        float volume_float(z, latitude, longitude) ;
                volume_float:valid_min = 3.f ;
                volume_float:valid_max = 59.f ;

// global attributes:
                :Conventions = "CF-1.5" ;
                :history = "GRASS GIS netCDF export of r3.out.netcdf" ;
data:

 longitude = -175, -165, -155, -145, -135, -125, -115, -105, -95, -85, -75,
    -65, -55, -45, -35, -25, -15, -5, 5, 15, 25, 35, 45, 55, 65, 75, 85, 95,
    105, 115, 125, 135, 145, 155, 165, 175 ;

 latitude = 85, 75, 65, 55, 45, 35, 25, 15, 5, -5, -15, -25, -35, -45, -55,
    -65, -75, -85 ;

 z = 0.5, 1.5, 2.5, 3.5, 4.5 ;
}

#####################################################################
# Exporting the projection specific settings and a null value
#####################################################################

r3.out.netcdf --o -p null=-1 input=volume_float output=test_float.nc
ncdump -c test_float.nc

netcdf test_float {
dimensions:
        longitude = 36 ;
        latitude = 18 ;
        z = 5 ;
variables:
        char crs ;
                crs:crs_wkt = "GEOGCS[\"wgs84\",DATUM[\"WGS_1984\",SPHEROID[\"WGS_1984\",6378137,298.257223563]],PRIMEM[\"Greenwich\",0],UNIT[\"degree\",0.0174532925199433]]" ;
                crs:spatial_ref = "GEOGCS[\"wgs84\",DATUM[\"WGS_1984\",SPHEROID[\"WGS_1984\",6378137,298.257223563]],PRIMEM[\"Greenwich\",0],UNIT[\"degree\",0.0174532925199433]]" ;
                crs:crs_proj4 = " +proj=longlat +no_defs +a=6378137 +rf=298.257223563 +towgs84=0.000,0.000,0.000" ;
        float longitude(longitude) ;
                longitude:units = "degrees_east" ;
                longitude:long_name = "Longitude values" ;
                longitude:standard_name = "longitude" ;
                longitude:axis = "X" ;
        float latitude(latitude) ;
                latitude:units = "degrees_north" ;
                latitude:long_name = "Latitude values" ;
                latitude:standard_name = "latitude" ;
                latitude:axis = "Y" ;
        float z(z) ;
                z:units = "meter" ;
                z:long_name = "z coordinate of projection" ;
                z:standard_name = "projection_z_coordinate" ;
                z:positive = "up" ;
                z:axis = "Z" ;
        float volume_float(z, latitude, longitude) ;
                volume_float:valid_min = 3.f ;
                volume_float:valid_max = 59.f ;
                volume_float:missing_value = -1.f ;
                volume_float:_FillValue = -1.f ;
                volume_float:grid_mapping = "crs" ;

// global attributes:
                :Conventions = "CF-1.5" ;
                :history = "GRASS GIS netCDF export of r3.out.netcdf" ;
data:

 longitude = -175, -165, -155, -145, -135, -125, -115, -105, -95, -85, -75,
    -65, -55, -45, -35, -25, -15, -5, 5, 15, 25, 35, 45, 55, 65, 75, 85, 95,
    105, 115, 125, 135, 145, 155, 165, 175 ;

 latitude = 85, 75, 65, 55, 45, 35, 25, 15, 5, -5, -15, -25, -35, -45, -55,
    -65, -75, -85 ;

 z = 0.5, 1.5, 2.5, 3.5, 4.5 ;
}

#####################################################################
# Assigning time as vertical unit and setting an absolute time stamp
#####################################################################

r3.timestamp map=volume_float date='1 Jan 2001/5 Jan 2001'
r3.support map=volume_float vunit="days"
r3.out.netcdf --o -p null=-1 input=volume_float output=test_float.nc
ncdump -c test_float.nc

netcdf test_float {
dimensions:
        longitude = 36 ;
        latitude = 18 ;
        time = 5 ;
variables:
        char crs ;
                crs:crs_wkt = "GEOGCS[\"wgs84\",DATUM[\"WGS_1984\",SPHEROID[\"WGS_1984\",6378137,298.257223563]],PRIMEM[\"Greenwich\",0],UNIT[\"degree\",0.0174532925199433]]" ;
                crs:spatial_ref = "GEOGCS[\"wgs84\",DATUM[\"WGS_1984\",SPHEROID[\"WGS_1984\",6378137,298.257223563]],PRIMEM[\"Greenwich\",0],UNIT[\"degree\",0.0174532925199433]]" ;
                crs:crs_proj4 = " +proj=longlat +no_defs +a=6378137 +rf=298.257223563 +towgs84=0.000,0.000,0.000" ;
        float longitude(longitude) ;
                longitude:units = "degrees_east" ;
                longitude:long_name = "Longitude values" ;
                longitude:standard_name = "longitude" ;
                longitude:axis = "X" ;
        float latitude(latitude) ;
                latitude:units = "degrees_north" ;
                latitude:long_name = "Latitude values" ;
                latitude:standard_name = "latitude" ;
                latitude:axis = "Y" ;
        int time(time) ;
                time:units = "days since 2001-01-01 00:00:00" ;
                time:long_name = "Time in days" ;
                time:calendar = "gregorian" ;
                time:positive = "up" ;
                time:axis = "T" ;
        float volume_float(time, latitude, longitude) ;
                volume_float:valid_min = 3.f ;
                volume_float:valid_max = 59.f ;
                volume_float:missing_value = -1.f ;
                volume_float:_FillValue = -1.f ;
                volume_float:grid_mapping = "crs" ;

// global attributes:
                :Conventions = "CF-1.5" ;
                :history = "GRASS GIS netCDF export of r3.out.netcdf" ;
data:

 longitude = -175, -165, -155, -145, -135, -125, -115, -105, -95, -85, -75,
    -65, -55, -45, -35, -25, -15, -5, 5, 15, 25, 35, 45, 55, 65, 75, 85, 95,
    105, 115, 125, 135, 145, 155, 165, 175 ;

 latitude = 85, 75, 65, 55, 45, 35, 25, 15, 5, -5, -15, -25, -35, -45, -55,
    -65, -75, -85 ;

 time = 0, 1, 2, 3, 4 ;
}

```

## SEE ALSO

*[t.rast.to.rast3](t.rast.to.rast3.md), [r3.in.ascii](r3.out.ascii.md),
[g.region](g.region.md)*

## AUTHORS

Sören Gebbert
