import json
from itertools import zip_longest

from grass.gunittest.case import TestCase
from grass.gunittest.main import test

from grass.gunittest.gmodules import SimpleModule


class TestVToDb(TestCase):
    @classmethod
    def setUpClass(cls):
        cls.use_temp_region()

    def _assert_dict_almost_equal(self, d1, d2):
        self.assertEqual(set(d1.keys()), set(d2.keys()))
        for k1 in d1:
            if isinstance(d1[k1], float):
                self.assertAlmostEqual(d1[k1], d2[k1], places=6)
            else:
                self.assertEqual(d1[k1], d2[k1])

    def _assert_json_equal(self, module, reference):
        self.runModule(module)
        result = json.loads(module.outputs.stdout)

        self.assertCountEqual(list(reference.keys()), list(result.keys()))
        self._assert_dict_almost_equal(reference["units"], result["units"])
        self._assert_dict_almost_equal(reference["totals"], result["totals"])
        for record1, record2 in zip_longest(reference["records"], result["records"]):
            self._assert_dict_almost_equal(record1, record2)

    def test_json_length(self):
        reference = {
            "units": {"length": "feet"},
            "totals": {"length": 34208.19507027471},
            "records": [
                {"category": 1, "length": 14944.03890742192},
                {"category": 2, "length": 19264.156162852792},
            ],
        }
        module = SimpleModule(
            "v.to.db",
            "busroute6",
            flags="p",
            option="length",
            units="feet",
            type="line",
            format="json",
        )
        self._assert_json_equal(module, reference)

    def test_json_coor(self):
        reference = {
            "units": {},
            "totals": {},
            "records": [
                {
                    "category": 11,
                    "x": 638150.7920150368,
                    "y": 220024.77915312737,
                    "z": 0,
                },
                {
                    "category": 103,
                    "x": 638287.18720843294,
                    "y": 219698.23416404743,
                    "z": 0,
                },
                {
                    "category": 104,
                    "x": 638278.98502463801,
                    "y": 219611.30807667322,
                    "z": 0,
                },
                {
                    "category": 105,
                    "x": 638306.74931247137,
                    "y": 219887.96339615693,
                    "z": 0,
                },
                {
                    "category": 106,
                    "x": 638269.46915021574,
                    "y": 219523.05001002364,
                    "z": 0,
                },
                {
                    "category": 107,
                    "x": 638232.74371348787,
                    "y": 219624.32918462454,
                    "z": 0,
                },
                {
                    "category": 108,
                    "x": 638230.51562103187,
                    "y": 219627.93192782634,
                    "z": 0,
                },
                {
                    "category": 109,
                    "x": 638262.64465328958,
                    "y": 219722.83464563562,
                    "z": 0,
                },
                {
                    "category": 110,
                    "x": 638260.06604013185,
                    "y": 219726.78790954105,
                    "z": 0,
                },
                {
                    "category": 111,
                    "x": 638495.58851612895,
                    "y": 219729.4579598321,
                    "z": 0,
                },
                {
                    "category": 112,
                    "x": 638332.230669952,
                    "y": 219728.48552823684,
                    "z": 0,
                },
                {
                    "category": 113,
                    "x": 638317.71299009491,
                    "y": 219988.20012615179,
                    "z": 0,
                },
                {
                    "category": 114,
                    "x": 638245.4447548897,
                    "y": 219808.26111248325,
                    "z": 0,
                },
                {
                    "category": 117,
                    "x": 638329.73432644235,
                    "y": 220099.23289130288,
                    "z": 0,
                },
            ],
        }
        module = SimpleModule(
            "v.to.db", "P079218", flags="p", option="coor", type="point", format="json"
        )
        self._assert_json_equal(module, reference)

    def test_json_count(self):
        reference = {
            "units": {},
            "totals": {"count": 2},
            "records": [{"category": 1, "count": 1}, {"category": 2, "count": 1}],
        }
        module = SimpleModule(
            "v.to.db",
            "busroute6",
            flags="p",
            option="count",
            type="line",
            format="json",
        )
        self._assert_json_equal(module, reference)

    def test_json_start(self):
        reference = {
            "units": {},
            "totals": {},
            "records": [
                {
                    "category": 1,
                    "x": 639023.64826068562,
                    "y": 226249.02352245309,
                    "z": 0,
                },
                {
                    "category": 2,
                    "x": 636004.4867560484,
                    "y": 227252.76225420492,
                    "z": 0,
                },
            ],
        }
        module = SimpleModule(
            "v.to.db",
            "busroute6",
            flags="p",
            option="start",
            type="line",
            format="json",
        )
        self._assert_json_equal(module, reference)

    def test_json_end(self):
        reference = {
            "units": {},
            "totals": {},
            "records": [
                {
                    "category": 1,
                    "x": 636004.4867560484,
                    "y": 227252.76225420492,
                    "z": 0,
                },
                {
                    "category": 2,
                    "x": 639023.64826068562,
                    "y": 226249.02352245309,
                    "z": 0,
                },
            ],
        }
        module = SimpleModule(
            "v.to.db",
            "busroute6",
            flags="p",
            option="end",
            type="line",
            format="json",
        )
        self._assert_json_equal(module, reference)

    def test_json_perimeter(self):
        reference = {
            "units": {"perimeter": "meters"},
            "totals": {},
            "records": [
                {"category": 1, "perimeter": 24598.86577244935},
                {"category": 2, "perimeter": 9468.4353925819196},
                {"category": 3, "perimeter": 35128.971279485551},
                {"category": 4, "perimeter": 11212.132236305841},
                {"category": 5, "perimeter": 9321.3918036502655},
                {"category": 6, "perimeter": 44835.796873804647},
                {"category": 7, "perimeter": 39073.911697927579},
                {"category": 8, "perimeter": 76857.730674167731},
                {"category": 9, "perimeter": 24498.427412088789},
                {"category": 10, "perimeter": 47137.575714678598},
                {"category": 11, "perimeter": 63352.593123459956},
                {"category": 12, "perimeter": 14528.794443747753},
                {"category": 13, "perimeter": 79233.504615285914},
                {"category": 14, "perimeter": 7804.7605523144985},
                {"category": 15, "perimeter": 10088.919010038735},
                {"category": 16, "perimeter": 42185.993420625222},
                {"category": 17, "perimeter": 59629.69189995146},
                {"category": 18, "perimeter": 38392.188493489128},
                {"category": 19, "perimeter": 16140.878937455964},
                {"category": 20, "perimeter": 6131.8264748468282},
                {"category": 21, "perimeter": 5722.2097281652468},
                {"category": 22, "perimeter": 981.84859206546867},
                {"category": 23, "perimeter": 3278.9333706111693},
                {"category": 24, "perimeter": 41171.96861221869},
                {"category": 25, "perimeter": 18430.430498643291},
                {"category": 26, "perimeter": 58923.003554658462},
                {"category": 27, "perimeter": 29549.526342969653},
                {"category": 28, "perimeter": 69802.55947781021},
                {"category": 29, "perimeter": 36058.290848741555},
                {"category": 30, "perimeter": 38449.518135314524},
                {"category": 31, "perimeter": 6250.7631358466524},
                {"category": 32, "perimeter": 55801.999859604068},
                {"category": 33, "perimeter": 70715.622869054729},
                {"category": 34, "perimeter": 76759.741107759342},
                {"category": 35, "perimeter": 26910.754068406088},
                {"category": 36, "perimeter": 39340.409485920383},
                {"category": 37, "perimeter": 145410.62931138012},
                {"category": 38, "perimeter": 50060.653495888575},
                {"category": 39, "perimeter": 55567.607184888642},
                {"category": 40, "perimeter": 87018.525604234936},
                {"category": 41, "perimeter": 51467.454846489643},
                {"category": 42, "perimeter": 64775.249413445606},
                {"category": 43, "perimeter": 82434.409033744654},
                {"category": 44, "perimeter": 5761.9738735064611},
            ],
        }
        module = SimpleModule(
            "v.to.db",
            "zipcodes",
            flags="p",
            option="perimeter",
            type="boundary",
            format="json",
        )
        self._assert_json_equal(module, reference)

    def test_json_area(self):
        reference = {
            "units": {"area": "square meters"},
            "totals": {"area": 2219442027.2203522},
            "records": [
                {"category": 1, "area": 24375323.127803534},
                {"category": 2, "area": 2938964.3204806102},
                {"category": 3, "area": 50536154.294107705},
                {"category": 4, "area": 5456815.4884436112},
                {"category": 5, "area": 1169066.1060796678},
                {"category": 6, "area": 55971618.79167185},
                {"category": 7, "area": 37359908.347983636},
                {"category": 8, "area": 122585299.61230192},
                {"category": 9, "area": 29156100.849060502},
                {"category": 10, "area": 58725489.008975707},
                {"category": 11, "area": 79361312.750253946},
                {"category": 12, "area": 8056494.6019629827},
                {"category": 13, "area": 110910429.39899969},
                {"category": 14, "area": 2368649.5703158271},
                {"category": 15, "area": 4548159.3697960936},
                {"category": 16, "area": 62407571.75144887},
                {"category": 17, "area": 101909096.69766694},
                {"category": 18, "area": 72939658.228119552},
                {"category": 19, "area": 6928400.7578192391},
                {"category": 20, "area": 1301839.0699078622},
                {"category": 21, "area": 1005940.9336275081},
                {"category": 22, "area": 55213.11751843011},
                {"category": 23, "area": 344611.97188792191},
                {"category": 24, "area": 56679125.811831549},
                {"category": 25, "area": 11981993.163624534},
                {"category": 26, "area": 76289561.70373407},
                {"category": 27, "area": 22538181.744323522},
                {"category": 28, "area": 129050077.06118551},
                {"category": 29, "area": 25640780.620766483},
                {"category": 30, "area": 30231334.130349141},
                {"category": 31, "area": 1559389.8359854589},
                {"category": 32, "area": 61973376.312294327},
                {"category": 33, "area": 84938883.430163413},
                {"category": 34, "area": 82929693.762826234},
                {"category": 35, "area": 30243435.083937138},
                {"category": 36, "area": 52403853.53241165},
                {"category": 37, "area": 260756049.76080063},
                {"category": 38, "area": 47323521.112179838},
                {"category": 39, "area": 51313713.890918314},
                {"category": 40, "area": 130875884.22338714},
                {"category": 41, "area": 59798016.861835025},
                {"category": 42, "area": 63169356.527421139},
                {"category": 43, "area": 98606087.58183229},
                {"category": 44, "area": 727592.90231111227},
            ],
        }
        module = SimpleModule(
            "v.to.db",
            "zipcodes",
            flags="p",
            option="area",
            type="centroid",
            columns="area_size",
            format="json",
        )
        self._assert_json_equal(module, reference)

    def test_json_fd(self):
        reference = {
            "units": {},
            "totals": {},
            "records": [
                {"category": 1, "fd": 1.1888302630715635},
                {"category": 2, "fd": 1.2294863225467458},
                {"category": 3, "fd": 1.1801402305121538},
                {"category": 4, "fd": 1.2022338481544725},
                {"category": 5, "fd": 1.3083671855769567},
                {"category": 6, "fd": 1.2007341724788474},
                {"category": 7, "fd": 1.2127947209115231},
                {"category": 8, "fd": 1.208066951839512},
                {"category": 9, "fd": 1.1759671238903131},
                {"category": 10, "fd": 1.2031076141730892},
                {"category": 11, "fd": 1.2156967493833235},
                {"category": 12, "fd": 1.2053696828117564},
                {"category": 13, "fd": 1.2178808404455326},
                {"category": 14, "fd": 1.2212280450720407},
                {"category": 15, "fd": 1.202746607648131},
                {"category": 16, "fd": 1.1866651678844309},
                {"category": 17, "fd": 1.1926412531635737},
                {"category": 18, "fd": 1.1660343427950151},
                {"category": 19, "fd": 1.2302742099545927},
                {"category": 20, "fd": 1.23887623246283},
                {"category": 21, "fd": 1.2519844727873344},
                {"category": 22, "fd": 1.2619223328747082},
                {"category": 23, "fd": 1.2698294530368857},
                {"category": 24, "fd": 1.1903391696095686},
                {"category": 25, "fd": 1.2052039170704054},
                {"category": 26, "fd": 1.2103535642862764},
                {"category": 27, "fd": 1.2159934303235393},
                {"category": 28, "fd": 1.1944311955007796},
                {"category": 29, "fd": 1.2301383257693426},
                {"category": 30, "fd": 1.2258317130564238},
                {"category": 31, "fd": 1.2258875533887401},
                {"category": 32, "fd": 1.2183071846625073},
                {"category": 33, "fd": 1.2232186165966428},
                {"category": 34, "fd": 1.2338205246210083},
                {"category": 35, "fd": 1.1843722268548442},
                {"category": 36, "fd": 1.1904709605382684},
                {"category": 37, "fd": 1.2268185301868757},
                {"category": 38, "fd": 1.2246122047934189},
                {"category": 39, "fd": 1.2307854820899107},
                {"category": 40, "fd": 1.2171238499654589},
                {"category": 41, "fd": 1.2117069453026579},
                {"category": 42, "fd": 1.2336145410258579},
                {"category": 43, "fd": 1.2299644069836664},
                {"category": 44, "fd": 1.2830579477351167},
            ],
        }
        module = SimpleModule(
            "v.to.db",
            "zipcodes",
            flags="p",
            option="fd",
            type="boundary",
            format="json",
        )
        self._assert_json_equal(module, reference)

    def test_json_compact(self):
        reference = {
            "units": {},
            "totals": {},
            "records": [
                {"category": 1, "compact": 3.6753703182945565},
                {"category": 2, "compact": 4.5652125910941628},
                {"category": 3, "compact": 4.1330860302272887},
                {"category": 4, "compact": 2.5515723117659221},
                {"category": 5, "compact": 3.4458068216072211},
                {"category": 6, "compact": 5.4223285951819333},
                {"category": 7, "compact": 2.474707367209521},
                {"category": 8, "compact": 1.7635412847652003},
                {"category": 9, "compact": 7.0107505994404562},
                {"category": 10, "compact": 4.8245329469298692},
                {"category": 11, "compact": 4.7932804440061689},
                {"category": 12, "compact": 4.1307727891921457},
                {"category": 13, "compact": 4.4839421485340925},
                {"category": 14, "compact": 3.0632422687208885},
                {"category": 15, "compact": 3.0090415115087539},
                {"category": 16, "compact": 2.6078537621385931},
                {"category": 17, "compact": 3.2889772616297641},
                {"category": 18, "compact": 2.417965657258295},
                {"category": 19, "compact": 4.0134126218532575},
                {"category": 20, "compact": 2.7796223999073204},
                {"category": 21, "compact": 2.79900443609572},
                {"category": 22, "compact": 2.2049195457558515},
                {"category": 23, "compact": 2.1068305760253234},
                {"category": 24, "compact": 2.2724659946920722},
                {"category": 25, "compact": 2.6002108310185501},
                {"category": 26, "compact": 3.1788654870378794},
                {"category": 27, "compact": 2.4181268737465977},
                {"category": 28, "compact": 2.3965831645238533},
                {"category": 29, "compact": 2.0780129123491937},
                {"category": 30, "compact": 4.8630796359296653},
                {"category": 31, "compact": 2.9823392332600531},
                {"category": 32, "compact": 2.3300076181728975},
                {"category": 33, "compact": 5.1000028977609313},
                {"category": 34, "compact": 5.920502404361156},
                {"category": 35, "compact": 5.5488403740189547},
                {"category": 36, "compact": 3.3907180136934643},
                {"category": 37, "compact": 2.5313870617681546},
                {"category": 38, "compact": 4.73821424080902},
                {"category": 39, "compact": 5.4916836681106842},
                {"category": 40, "compact": 2.2745474591353347},
                {"category": 41, "compact": 2.9809652471754351},
                {"category": 42, "compact": 4.6962590533765969},
                {"category": 43, "compact": 2.1245231830346061},
                {"category": 44, "compact": 3.056012063621615},
                {"category": 45, "compact": 3.3291343464190493},
                {"category": 46, "compact": 5.1933032154109879},
                {"category": 47, "compact": 4.1796131665762353},
                {"category": 48, "compact": 5.2623139999057207},
                {"category": 49, "compact": 1.9600793652891744},
                {"category": 50, "compact": 2.5171374284117043},
                {"category": 51, "compact": 6.2768503000458189},
                {"category": 52, "compact": 4.2431120825068325},
                {"category": 53, "compact": 3.9101032574787369},
                {"category": 54, "compact": 8.9677872869774795},
                {"category": 55, "compact": 6.9027292967097438},
                {"category": 56, "compact": 4.3365111677624961},
                {"category": 57, "compact": 4.0262796900466329},
                {"category": 58, "compact": 7.2958538755198319},
                {"category": 59, "compact": 5.5278721435645668},
                {"category": 60, "compact": 2.5869340127782872},
                {"category": 61, "compact": 6.0408939089999301},
                {"category": 62, "compact": 4.8365780371494163},
                {"category": 63, "compact": 3.917141398042034},
                {"category": 64, "compact": 4.1301496064039123},
                {"category": 65, "compact": 2.8273276432355363},
                {"category": 66, "compact": 2.5923462660986591},
                {"category": 67, "compact": 4.4259345650251367},
                {"category": 68, "compact": 3.2137515594108756},
                {"category": 69, "compact": 5.1014676630159501},
                {"category": 70, "compact": 3.4286661194216475},
                {"category": 71, "compact": 5.2859091659363777},
                {"category": 72, "compact": 3.6359093682608199},
                {"category": 73, "compact": 5.4962989484868796},
                {"category": 74, "compact": 11.452610638247759},
                {"category": 75, "compact": 5.0088687194692012},
                {"category": 76, "compact": 4.0467321897356303},
                {"category": 77, "compact": 4.0089480465834093},
                {"category": 78, "compact": 2.3520946727425192},
                {"category": 79, "compact": 2.6407754739017859},
                {"category": 80, "compact": 2.0934143925538549},
                {"category": 81, "compact": 4.4526475376702024},
                {"category": 82, "compact": 4.6351273439475236},
                {"category": 83, "compact": 1.7158259909427274},
                {"category": 84, "compact": 4.4541759550429267},
                {"category": 85, "compact": 3.5790482554384973},
                {"category": 86, "compact": 5.7877762918161766},
                {"category": 87, "compact": 2.371047292564501},
                {"category": 88, "compact": 7.9942310159547709},
                {"category": 89, "compact": 2.3153381229207834},
                {"category": 90, "compact": 3.6342338670560279},
                {"category": 91, "compact": 2.6648488853708723},
                {"category": 92, "compact": 3.2259779985800883},
                {"category": 93, "compact": 4.8501132588513514},
                {"category": 94, "compact": 4.1743562627031441},
                {"category": 95, "compact": 3.2024459554511591},
                {"category": 96, "compact": 2.3924837573909778},
                {"category": 97, "compact": 3.8614158957079829},
                {"category": 98, "compact": 3.7644090019690868},
                {"category": 99, "compact": 2.7647815274299017},
                {"category": 100, "compact": 1.6069454031200812},
                {"category": 101, "compact": 2.745207032999478},
                {"category": 102, "compact": 3.9012908531821631},
                {"category": 103, "compact": 1.6961214333217043},
                {"category": 104, "compact": 2.9654999593207316},
                {"category": 105, "compact": 5.1303074756554601},
                {"category": 106, "compact": 4.1971143513731475},
                {"category": 107, "compact": 3.0189983714399524},
                {"category": 108, "compact": 4.5821868837403565},
            ],
        }
        module = SimpleModule(
            "v.to.db",
            "urbanarea",
            flags="p",
            option="compact",
            type="point",
            format="json",
        )
        self._assert_json_equal(module, reference)

    def test_json_azimuth(self):
        reference = {
            "units": {"azimuth": "radians"},
            "totals": {},
            "records": [
                {"category": 1, "azimuth": 0.15552736300202469},
                {"category": 2, "azimuth": 6.2611444061916544},
                {"category": 3, "azimuth": 4.7113927787366947},
                {"category": 4, "azimuth": 6.0610740057486403},
                {"category": 5, "azimuth": 1.3834383508604806},
                {"category": 6, "azimuth": 0.52165746607998187},
                {"category": 7, "azimuth": 0.48360687184449103},
                {"category": 8, "azimuth": 4.3873203129015987},
                {"category": 9, "azimuth": 4.917994160395911},
                {"category": 10, "azimuth": 4.7054516640686934},
                {"category": 11, "azimuth": 4.7545566810126143},
                {"category": 12, "azimuth": 4.6689095777115579},
                {"category": 13, "azimuth": 1.9556156580016921},
                {"category": 14, "azimuth": 3.4603499315931314},
                {"category": 15, "azimuth": 1.3779814182438492},
                {"category": 16, "azimuth": 5.8073067702699781},
                {"category": 17, "azimuth": 1.7046448759630304},
                {"category": 18, "azimuth": 0.048153855972286226},
                {"category": 19, "azimuth": 0.70409134237079107},
                {"category": 20, "azimuth": 5.6657069067001844},
                {"category": 21, "azimuth": 3.8673913247832115},
                {"category": 22, "azimuth": 6.2688653187448447},
                {"category": 23, "azimuth": 0.23544541431479746},
                {"category": 24, "azimuth": 0.81663440548732902},
                {"category": 25, "azimuth": 5.5569009519576467},
                {"category": 26, "azimuth": 6.1808883708660849},
                {"category": 27, "azimuth": 0.34875780607993206},
                {"category": 28, "azimuth": 0.86096427065887937},
                {"category": 29, "azimuth": 5.4257798050382338},
                {"category": 30, "azimuth": 5.5027843259499916},
                {"category": 31, "azimuth": 5.4094300185811681},
                {"category": 32, "azimuth": 1.0632713608135},
                {"category": 33, "azimuth": 6.2061517525817917},
                {"category": 34, "azimuth": 0.47121463176820988},
                {"category": 35, "azimuth": 5.8370081157361806},
                {"category": 36, "azimuth": 5.4454100448648131},
                {"category": 37, "azimuth": 5.4165401994734825},
                {"category": 38, "azimuth": 0.65097277948601195},
                {"category": 39, "azimuth": 2.6611745267457536},
                {"category": 40, "azimuth": 6.1861549032630485},
                {"category": 41, "azimuth": 3.0445622496732554},
                {"category": 42, "azimuth": 0.16243481801048862},
                {"category": 43, "azimuth": 0.077777060385833319},
                {"category": 44, "azimuth": 0.077777060385833319},
                {"category": 45, "azimuth": 2.9175565430851629},
                {"category": 46, "azimuth": 2.6625696599897539},
                {"category": 47, "azimuth": 5.804162313579547},
                {"category": 48, "azimuth": 5.7471453409697322},
                {"category": 49, "azimuth": 2.6055526873799391},
                {"category": 50, "azimuth": 0.017634564254405721},
                {"category": 51, "azimuth": 5.363538499432293},
                {"category": 52, "azimuth": 5.3840655693972819},
                {"category": 53, "azimuth": 0.85841667826683066},
                {"category": 54, "azimuth": 0.1116607111701752},
                {"category": 55, "azimuth": 0.76557730422293069},
                {"category": 56, "azimuth": 5.6739644955163149},
                {"category": 57, "azimuth": 5.4663436428888481},
                {"category": 58, "azimuth": 0.73244235081293119},
                {"category": 59, "azimuth": 0.73235063382396415},
                {"category": 60, "azimuth": 0.73303517198856671},
                {"category": 61, "azimuth": 4.8639234522440953},
                {"category": 62, "azimuth": 1.1989344314913484},
                {"category": 63, "azimuth": 1.2744026553731422},
                {"category": 64, "azimuth": 1.2046386684920709},
                {"category": 65, "azimuth": 4.4032442853407128},
                {"category": 66, "azimuth": 1.6198952775747559},
                {"category": 67, "azimuth": 1.6060697028170681},
                {"category": 68, "azimuth": 5.9628917697198141},
                {"category": 69, "azimuth": 1.8960717288553166},
                {"category": 70, "azimuth": 5.7752995253508317},
                {"category": 71, "azimuth": 5.7752995253508317},
                {"category": 72, "azimuth": 5.7748746822182619},
                {"category": 73, "azimuth": 1.6334602093393629},
                {"category": 74, "azimuth": 5.3750304049691202},
                {"category": 75, "azimuth": 1.244266373284256},
                {"category": 76, "azimuth": 1.609786309423533},
                {"category": 77, "azimuth": 1.6148201054431088},
                {"category": 78, "azimuth": 4.5214801165837581},
                {"category": 79, "azimuth": 1.4684925040918633},
                {"category": 80, "azimuth": 1.5774527255019386},
                {"category": 81, "azimuth": 1.2184560052906659},
                {"category": 82, "azimuth": 5.3157510170940094},
                {"category": 83, "azimuth": 1.1739085343633997},
                {"category": 84, "azimuth": 0.51165480047264966},
                {"category": 85, "azimuth": 1.3894109258751763},
                {"category": 86, "azimuth": 6.2330271931594456},
                {"category": 87, "azimuth": 1.9867738533090904},
                {"category": 88, "azimuth": 1.7349669250062523},
                {"category": 89, "azimuth": 1.3921696052156096},
                {"category": 90, "azimuth": 5.7753235337983897},
                {"category": 91, "azimuth": 1.7821758213213765},
                {"category": 92, "azimuth": 4.6074061543331162},
                {"category": 93, "azimuth": 1.0727762646416135},
                {"category": 94, "azimuth": 3.2958445544800701},
                {"category": 95, "azimuth": 1.7052386521537859},
                {"category": 96, "azimuth": 1.6714057806433511},
                {"category": 97, "azimuth": 2.0468559608406514},
                {"category": 98, "azimuth": 2.1430983222979161},
                {"category": 99, "azimuth": 0.45765764160362132},
                {"category": 100, "azimuth": 1.3644636128977239},
                {"category": 101, "azimuth": 5.0983965164753826},
                {"category": 102, "azimuth": 1.345230605931621},
                {"category": 103, "azimuth": 2.1988352288590969},
                {"category": 104, "azimuth": 1.3455770885022402},
                {"category": 105, "azimuth": 2.6111253128578702},
                {"category": 106, "azimuth": 2.2503194327076859},
                {"category": 107, "azimuth": 1.9163630218199224},
                {"category": 108, "azimuth": 5.8762915215625195},
                {"category": 109, "azimuth": 2.2762260194486985},
                {"category": 110, "azimuth": 5.7792222682523686},
                {"category": 111, "azimuth": 1.2719499241846048},
                {"category": 112, "azimuth": 5.7800150680037605},
                {"category": 113, "azimuth": 1.23784802487287},
                {"category": 114, "azimuth": 6.2257028109937167},
                {"category": 115, "azimuth": 4.6266653843300345},
                {"category": 116, "azimuth": 2.0657907494432033},
                {"category": 117, "azimuth": 1.860610786911868},
                {"category": 118, "azimuth": 5.8540572515966041},
                {"category": 119, "azimuth": 4.2250810437886068},
                {"category": 120, "azimuth": 6.259374556664925},
                {"category": 121, "azimuth": 1.8269730752668589},
                {"category": 122, "azimuth": 5.2749527481117404},
                {"category": 123, "azimuth": 1.8237545086211651},
                {"category": 124, "azimuth": 1.8236631758715891},
                {"category": 125, "azimuth": 6.2592267587548491},
                {"category": 126, "azimuth": 2.1374132289381773},
                {"category": 127, "azimuth": 5.2776173179677368},
                {"category": 128, "azimuth": 1.460336513003101},
                {"category": 129, "azimuth": 6.2727187006740799},
                {"category": 130, "azimuth": 4.6394501294459358},
                {"category": 131, "azimuth": 1.8197850760937411},
                {"category": 132, "azimuth": 5.302426037015044},
                {"category": 133, "azimuth": 5.2563984849755716},
                {"category": 134, "azimuth": 2.3270636603786938},
                {"category": 135, "azimuth": 5.4686563139684869},
                {"category": 136, "azimuth": 2.2464632005619958},
                {"category": 137, "azimuth": 3.7195869351846333},
                {"category": 138, "azimuth": 5.6655556306914265},
                {"category": 139, "azimuth": 0.65060216913678215},
                {"category": 140, "azimuth": 4.955394147349276},
                {"category": 141, "azimuth": 3.7197530552788831},
                {"category": 142, "azimuth": 2.4713088908867937},
                {"category": 143, "azimuth": 1.0295261852168418},
                {"category": 144, "azimuth": 4.4046906225176068},
                {"category": 145, "azimuth": 4.4214391308889507},
                {"category": 146, "azimuth": 1.2798464772991578},
                {"category": 147, "azimuth": 2.5643403240910669},
                {"category": 148, "azimuth": 1.1312185171127314},
                {"category": 149, "azimuth": 6.1516437232872816},
                {"category": 150, "azimuth": 2.1292843414100062},
                {"category": 151, "azimuth": 1.1537180555443243},
                {"category": 152, "azimuth": 2.5640442866165247},
                {"category": 153, "azimuth": 2.56428050613891},
                {"category": 154, "azimuth": 3.281835775702441},
                {"category": 155, "azimuth": 6.1508774944144973},
                {"category": 156, "azimuth": 2.5643076700172056},
                {"category": 157, "azimuth": 6.2145220874116047},
                {"category": 158, "azimuth": 5.4680578397976021},
                {"category": 159, "azimuth": 2.0389711413034979},
                {"category": 160, "azimuth": 1.2524518453606917},
                {"category": 161, "azimuth": 0.10357239264643422},
                {"category": 162, "azimuth": 0.18221234493356422},
                {"category": 163, "azimuth": 3.1416344229389401},
                {"category": 164, "azimuth": 0.55783608161167642},
                {"category": 165, "azimuth": 6.2493903496156165},
                {"category": 166, "azimuth": 5.7758587197688698},
                {"category": 167, "azimuth": 1.0931499553351052},
                {"category": 168, "azimuth": 3.8198623969492855},
                {"category": 169, "azimuth": 1.5341428500654815},
                {"category": 170, "azimuth": 3.9898547916487228},
                {"category": 171, "azimuth": 1.8675262194314179},
                {"category": 172, "azimuth": 4.7442882277697063},
                {"category": 173, "azimuth": 1.7029286733093032},
                {"category": 174, "azimuth": 4.8445213268990965},
                {"category": 175, "azimuth": 6.2614297626751245},
                {"category": 176, "azimuth": 4.9440054097625623},
                {"category": 177, "azimuth": 4.9438777367923308},
                {"category": 178, "azimuth": 1.8018178679596342},
                {"category": 179, "azimuth": 4.9434105215494268},
                {"category": 180, "azimuth": 6.2678448091347825},
                {"category": 181, "azimuth": 4.1996485792464293},
                {"category": 182, "azimuth": 5.9299397766378519},
                {"category": 183, "azimuth": 5.023213219340974},
                {"category": 184, "azimuth": 1.6613562313840651},
                {"category": 185, "azimuth": 5.006851025022514},
                {"category": 186, "azimuth": 4.9434105215494268},
                {"category": 187, "azimuth": 0.45824681116307159},
                {"category": 188, "azimuth": 0.0094260607047866787},
                {"category": 189, "azimuth": 2.2677650396787739},
                {"category": 190, "azimuth": 1.409553654098656},
                {"category": 191, "azimuth": 0.12388498896597039},
                {"category": 192, "azimuth": 2.3936405889877248},
                {"category": 193, "azimuth": 5.5165725024275236},
                {"category": 194, "azimuth": 4.9656425270649365},
                {"category": 195, "azimuth": 4.0239416036562972},
                {"category": 196, "azimuth": 4.7999388973924146},
                {"category": 197, "azimuth": 0.2603642715632089},
                {"category": 198, "azimuth": 1.7419877029833255},
                {"category": 199, "azimuth": 4.8835803565731188},
                {"category": 200, "azimuth": 0.24168035911722505},
                {"category": 201, "azimuth": 1.480002663058176},
                {"category": 202, "azimuth": 1.7684828041738077},
                {"category": 203, "azimuth": 5.9757125821090975},
                {"category": 204, "azimuth": 0.45408641009041473},
                {"category": 205, "azimuth": 6.0697992702116048},
                {"category": 206, "azimuth": 0.93062352523229541},
                {"category": 207, "azimuth": 2.2747077120057639},
                {"category": 208, "azimuth": 3.9723545545849275},
                {"category": 209, "azimuth": 5.946073088628995},
                {"category": 210, "azimuth": 2.2761500533377257},
                {"category": 211, "azimuth": 1.5831617062378838},
                {"category": 212, "azimuth": 0.45794931459193039},
                {"category": 213, "azimuth": 1.4547142040495664},
                {"category": 214, "azimuth": 2.1111503777681908},
                {"category": 215, "azimuth": 0.45907447015684055},
                {"category": 216, "azimuth": 2.0797417216703793},
                {"category": 217, "azimuth": 0.45843932269413207},
                {"category": 218, "azimuth": 2.0793139228046047},
                {"category": 219, "azimuth": 0.15674419028458825},
                {"category": 220, "azimuth": 5.222393827486826},
                {"category": 221, "azimuth": 4.7992940350536166},
                {"category": 222, "azimuth": 5.1191472099802464},
                {"category": 223, "azimuth": 0.88190371057993167},
                {"category": 224, "azimuth": 0.49373281562924776},
                {"category": 225, "azimuth": 0.51445131308387204},
                {"category": 226, "azimuth": 3.2500838131701637},
                {"category": 227, "azimuth": 4.6028979326087756},
                {"category": 228, "azimuth": 6.1761959267223974},
                {"category": 229, "azimuth": 3.8727976709003014},
                {"category": 230, "azimuth": 5.2400827697662127},
                {"category": 231, "azimuth": 3.6674316623526044},
                {"category": 232, "azimuth": 2.909706341286479},
                {"category": 233, "azimuth": 5.9506327175177516},
                {"category": 234, "azimuth": 4.2307626644819729},
                {"category": 235, "azimuth": 5.0685969960951995},
                {"category": 236, "azimuth": 5.6869470061059486},
                {"category": 237, "azimuth": 5.7482121642313837},
                {"category": 238, "azimuth": 1.1613320281881421},
                {"category": 239, "azimuth": 1.8775916805919488},
                {"category": 240, "azimuth": 1.1188910705392934},
                {"category": 241, "azimuth": 5.7615128825669739},
                {"category": 242, "azimuth": 5.7542656444323148},
                {"category": 243, "azimuth": 5.7661008882280935},
                {"category": 244, "azimuth": 5.857930701638046},
                {"category": 245, "azimuth": 3.3810382319782137},
                {"category": 246, "azimuth": 5.8083857441522211},
                {"category": 247, "azimuth": 2.7416344884058756},
                {"category": 248, "azimuth": 1.107547839706466},
                {"category": 249, "azimuth": 3.098349281833205},
                {"category": 250, "azimuth": 1.0396603794675703},
                {"category": 251, "azimuth": 0.48575222089003139},
                {"category": 252, "azimuth": 5.9076920387658278},
                {"category": 253, "azimuth": 5.9448616847168303},
                {"category": 254, "azimuth": 1.9999932042375017},
                {"category": 255, "azimuth": 5.0938664691636575},
                {"category": 256, "azimuth": 5.0938664691636575},
                {"category": 257, "azimuth": 4.7415254644172062},
                {"category": 258, "azimuth": 4.8428285315398334},
                {"category": 259, "azimuth": 4.8197922806238296},
                {"category": 260, "azimuth": 4.7313276526433254},
                {"category": 261, "azimuth": 3.6555576704195887},
                {"category": 262, "azimuth": 4.463625642281122},
                {"category": 263, "azimuth": 4.8801136949065169},
                {"category": 264, "azimuth": 3.7481040016446641},
                {"category": 265, "azimuth": 2.2512279212552517},
                {"category": 266, "azimuth": 3.1010528408965272},
                {"category": 267, "azimuth": 6.2426454944863199},
                {"category": 268, "azimuth": 5.0159779228307766},
                {"category": 269, "azimuth": 4.7289751369643502},
                {"category": 270, "azimuth": 4.5118352941347197},
                {"category": 271, "azimuth": 4.6454589577880281},
                {"category": 272, "azimuth": 1.0243514724091918},
                {"category": 273, "azimuth": 4.6430655123597306},
                {"category": 274, "azimuth": 3.4977443486716964},
                {"category": 275, "azimuth": 3.1009930124457101},
                {"category": 276, "azimuth": 2.1750503594042572},
                {"category": 277, "azimuth": 5.0346489121450855},
                {"category": 278, "azimuth": 0.98444767597152416},
                {"category": 279, "azimuth": 3.1085994190990469},
                {"category": 280, "azimuth": 0.86937783467351748},
                {"category": 281, "azimuth": 6.0904890533534859},
                {"category": 282, "azimuth": 2.8556850936805493},
                {"category": 283, "azimuth": 2.8556850936805493},
                {"category": 284, "azimuth": 0.084612675972327522},
                {"category": 285, "azimuth": 6.1668022132410947},
                {"category": 286, "azimuth": 5.5780902737887379},
                {"category": 287, "azimuth": 4.764785010592739},
                {"category": 288, "azimuth": 4.5998026256588922},
                {"category": 289, "azimuth": 4.5883781229789582},
                {"category": 290, "azimuth": 0.9827738988254473},
                {"category": 291, "azimuth": 4.986858839382478},
                {"category": 292, "azimuth": 4.4129107715259437},
                {"category": 293, "azimuth": 1.9013746579457651},
                {"category": 294, "azimuth": 1.7764226911000049},
                {"category": 295, "azimuth": 1.0120813938193858},
                {"category": 296, "azimuth": 1.7847865233122315},
                {"category": 297, "azimuth": 1.0081464561184024},
                {"category": 298, "azimuth": 6.004662712577824},
                {"category": 299, "azimuth": 1.005661210333278},
                {"category": 300, "azimuth": 3.3225791215644014},
                {"category": 301, "azimuth": 5.6676567382610434},
                {"category": 302, "azimuth": 3.1716763758957236},
                {"category": 303, "azimuth": 2.8441206564442583},
                {"category": 304, "azimuth": 1.0715345726475995},
                {"category": 305, "azimuth": 0.73099644957657872},
                {"category": 306, "azimuth": 4.0376356775234719},
                {"category": 307, "azimuth": 5.4931107312392253},
                {"category": 308, "azimuth": 0.67767125312399557},
                {"category": 309, "azimuth": 2.7343009068280928},
                {"category": 310, "azimuth": 0.47500354768183123},
                {"category": 311, "azimuth": 2.7344097330969155},
                {"category": 312, "azimuth": 0.68699836501948741},
                {"category": 313, "azimuth": 3.6136587131603659},
                {"category": 314, "azimuth": 0.97823549192748604},
                {"category": 315, "azimuth": 0.82664459852539374},
                {"category": 316, "azimuth": 2.1690029369658341},
                {"category": 317, "azimuth": 2.8522943949596371},
                {"category": 318, "azimuth": 5.0576033962105331},
                {"category": 319, "azimuth": 3.6854971358865751},
                {"category": 320, "azimuth": 4.1527374692417807},
                {"category": 321, "azimuth": 1.9377160140587424},
                {"category": 322, "azimuth": 2.9882647347205196},
                {"category": 323, "azimuth": 4.1629624853074141},
                {"category": 324, "azimuth": 2.0733576698686988},
                {"category": 325, "azimuth": 2.9697544067095958},
                {"category": 326, "azimuth": 3.7631302145038013},
                {"category": 327, "azimuth": 2.904237079403964},
                {"category": 328, "azimuth": 4.1796813906156896},
                {"category": 329, "azimuth": 4.2315712051328305},
                {"category": 330, "azimuth": 3.1007889314404098},
                {"category": 331, "azimuth": 3.246426837276768},
                {"category": 332, "azimuth": 3.5180036522736815},
                {"category": 333, "azimuth": 3.349870957053112},
                {"category": 334, "azimuth": 2.5455118970866955},
                {"category": 335, "azimuth": 2.7746292257664731},
                {"category": 336, "azimuth": 3.9415471820923385},
                {"category": 337, "azimuth": 4.1033825844909009},
                {"category": 338, "azimuth": 1.9260296862814419},
                {"category": 339, "azimuth": 2.5388157211050317},
                {"category": 340, "azimuth": 1.5056022240985119},
                {"category": 341, "azimuth": 1.3449124010920286},
                {"category": 342, "azimuth": 1.6904122882388359},
                {"category": 343, "azimuth": 6.0974139871805386},
                {"category": 344, "azimuth": 1.5259924839992625},
                {"category": 345, "azimuth": 1.5953219279030557},
                {"category": 346, "azimuth": 1.3862154960481416},
                {"category": 347, "azimuth": 0.65556586107989179},
                {"category": 348, "azimuth": 1.8101876555152849},
                {"category": 349, "azimuth": 4.439972214226521},
                {"category": 350, "azimuth": 4.1700364878493588},
                {"category": 351, "azimuth": 6.1764586470109126},
                {"category": 352, "azimuth": 3.4391032341761818},
                {"category": 353, "azimuth": 6.0126023912944051},
                {"category": 354, "azimuth": 1.7460951510552212},
                {"category": 355, "azimuth": 4.1125239546585313},
            ],
        }
        module = SimpleModule(
            "v.to.db",
            "roadsmajor@PERMANENT",
            flags="p",
            option="azimuth",
            type="line",
            units="radians",
            format="json",
        )
        self._assert_json_equal(module, reference)

        reference_deg = {
            "units": {"azimuth": "degrees"},
            "totals": {},
            "records": [
                {"category": 1, "azimuth": 8.9110614988151244},
                {"category": 2, "azimuth": 358.73714939672578},
                {"category": 3, "azimuth": 269.94292185002593},
                {"category": 4, "azimuth": 347.27395984584876},
                {"category": 5, "azimuth": 79.265178720844318},
                {"category": 6, "azimuth": 29.888771157871862},
                {"category": 7, "azimuth": 27.708632700213418},
                {"category": 8, "azimuth": 251.37493730127736},
                {"category": 9, "azimuth": 281.78030906067056},
                {"category": 10, "azimuth": 269.60252105394619},
                {"category": 11, "azimuth": 272.41603127775124},
                {"category": 12, "azimuth": 267.50881373107973},
                {"category": 13, "azimuth": 112.04852355319636},
                {"category": 14, "azimuth": 198.26344671866957},
                {"category": 15, "azimuth": 78.952519512824054},
                {"category": 16, "azimuth": 332.73416827421886},
                {"category": 17, "azimuth": 97.668956961283357},
                {"category": 18, "azimuth": 2.759012714492834},
                {"category": 19, "azimuth": 40.341462309547005},
                {"category": 20, "azimuth": 324.62109371204144},
                {"category": 21, "azimuth": 221.58520063558623},
                {"category": 22, "azimuth": 359.17952510001317},
                {"category": 23, "azimuth": 13.490028545946952},
                {"category": 24, "azimuth": 46.789704839599068},
                {"category": 25, "azimuth": 318.3869717194026},
                {"category": 26, "azimuth": 354.13881729211778},
                {"category": 27, "azimuth": 19.98235036062211},
                {"category": 28, "azimuth": 49.329619020312883},
                {"category": 29, "azimuth": 310.87428339600541},
                {"category": 30, "azimuth": 315.28631744767603},
                {"category": 31, "azimuth": 309.93750963607545},
                {"category": 32, "azimuth": 60.920961451745299},
                {"category": 33, "azimuth": 355.5863024406558},
                {"category": 34, "azimuth": 26.99860964512963},
                {"category": 35, "azimuth": 334.43593001529229},
                {"category": 36, "azimuth": 311.99901328889803},
                {"category": 37, "azimuth": 310.34489299277959},
                {"category": 38, "azimuth": 37.297992842448899},
                {"category": 39, "azimuth": 152.4740689302559},
                {"category": 40, "azimuth": 354.44056737113272},
                {"category": 41, "azimuth": 174.44056737113274},
                {"category": 42, "azimuth": 9.3068295179766096},
                {"category": 43, "azimuth": 4.4562973030423958},
                {"category": 44, "azimuth": 4.4562973030423958},
                {"category": 45, "azimuth": 167.16367640955815},
                {"category": 46, "azimuth": 152.55400417699551},
                {"category": 47, "azimuth": 332.55400417699553},
                {"category": 48, "azimuth": 329.28717228584009},
                {"category": 49, "azimuth": 149.28717228584011},
                {"category": 50, "azimuth": 1.0103861053297132},
                {"category": 51, "azimuth": 307.30811927340108},
                {"category": 52, "azimuth": 308.48423374816468},
                {"category": 53, "azimuth": 49.183652728328859},
                {"category": 54, "azimuth": 6.3976874874803267},
                {"category": 55, "azimuth": 43.864348422976988},
                {"category": 56, "azimuth": 325.09421870016013},
                {"category": 57, "azimuth": 313.19842010569863},
                {"category": 58, "azimuth": 41.965855438221396},
                {"category": 59, "azimuth": 41.960600441843937},
                {"category": 60, "azimuth": 41.999821589591299},
                {"category": 61, "azimuth": 278.68228568828789},
                {"category": 62, "azimuth": 68.693882837371007},
                {"category": 63, "azimuth": 73.017893553146195},
                {"category": 64, "azimuth": 69.02071154285477},
                {"category": 65, "azimuth": 252.28731371512123},
                {"category": 66, "azimuth": 92.813162658206508},
                {"category": 67, "azimuth": 92.021015575248384},
                {"category": 68, "azimuth": 341.64853209823974},
                {"category": 69, "azimuth": 108.63690771748303},
                {"category": 70, "azimuth": 330.90028822651027},
                {"category": 71, "azimuth": 330.90028822651027},
                {"category": 72, "azimuth": 330.87594650805886},
                {"category": 73, "azimuth": 93.590375997701429},
                {"category": 74, "azimuth": 307.96655695922431},
                {"category": 75, "azimuth": 71.291211779237315},
                {"category": 76, "azimuth": 92.233961447909266},
                {"category": 77, "azimuth": 92.522376714760711},
                {"category": 78, "azimuth": 259.06172783256875},
                {"category": 79, "azimuth": 84.138422731061539},
                {"category": 80, "azimuth": 90.381383552669845},
                {"category": 81, "azimuth": 69.812386625525065},
                {"category": 82, "azimuth": 304.57009822186149},
                {"category": 83, "azimuth": 67.26000455341098},
                {"category": 84, "azimuth": 29.315660634691064},
                {"category": 85, "azimuth": 79.607382062011666},
                {"category": 86, "azimuth": 357.12615175830996},
                {"category": 87, "azimuth": 113.8337566415546},
                {"category": 88, "azimuth": 99.406282397648667},
                {"category": 89, "azimuth": 79.765442745248436},
                {"category": 90, "azimuth": 330.90166380922795},
                {"category": 91, "azimuth": 102.11115291197599},
                {"category": 92, "azimuth": 263.98492714588878},
                {"category": 93, "azimuth": 61.465552325773935},
                {"category": 94, "azimuth": 188.83798290288314},
                {"category": 95, "azimuth": 97.702977830988999},
                {"category": 96, "azimuth": 95.76449708463268},
                {"category": 97, "azimuth": 117.27620782736423},
                {"category": 98, "azimuth": 122.79048894923804},
                {"category": 99, "azimuth": 26.22185132579834},
                {"category": 100, "azimuth": 78.1780063182117},
                {"category": 101, "azimuth": 292.1166026782405},
                {"category": 102, "azimuth": 77.076036191708297},
                {"category": 103, "azimuth": 125.98397845830873},
                {"category": 104, "azimuth": 77.095888180679609},
                {"category": 105, "azimuth": 149.60646020653263},
                {"category": 106, "azimuth": 128.93380605042407},
                {"category": 107, "azimuth": 109.79951316521844},
                {"category": 108, "azimuth": 336.68670337404114},
                {"category": 109, "azimuth": 130.41814413227365},
                {"category": 110, "azimuth": 331.12504483888324},
                {"category": 111, "azimuth": 72.877362407762902},
                {"category": 112, "azimuth": 331.17046891863697},
                {"category": 113, "azimuth": 70.923467503820405},
                {"category": 114, "azimuth": 356.70649557267285},
                {"category": 115, "azimuth": 265.08839974138397},
                {"category": 116, "azimuth": 118.36109130026287},
                {"category": 117, "azimuth": 106.60514540656499},
                {"category": 118, "azimuth": 335.41277354443974},
                {"category": 119, "azimuth": 242.07931190981574},
                {"category": 120, "azimuth": 358.63574448847095},
                {"category": 121, "azimuth": 104.67784649682791},
                {"category": 122, "azimuth": 302.23252959773794},
                {"category": 123, "azimuth": 104.49343621194807},
                {"category": 124, "azimuth": 104.48820323086603},
                {"category": 125, "azimuth": 358.62727629200276},
                {"category": 126, "azimuth": 122.46475709358715},
                {"category": 127, "azimuth": 302.38519820470435},
                {"category": 128, "azimuth": 83.671118863929152},
                {"category": 129, "azimuth": 359.40030762141032},
                {"category": 130, "azimuth": 265.8209116786756},
                {"category": 131, "azimuth": 104.26600448106473},
                {"category": 132, "azimuth": 303.80663310124083},
                {"category": 133, "azimuth": 301.16944862806031},
                {"category": 134, "azimuth": 133.33092639796394},
                {"category": 135, "azimuth": 313.33092639796394},
                {"category": 136, "azimuth": 128.71286022365334},
                {"category": 137, "azimuth": 213.11663291808037},
                {"category": 138, "azimuth": 324.61242623519803},
                {"category": 139, "azimuth": 37.276758433594161},
                {"category": 140, "azimuth": 283.92317046694268},
                {"category": 141, "azimuth": 213.1261508983732},
                {"category": 142, "azimuth": 141.59556932096976},
                {"category": 143, "azimuth": 58.98750531112892},
                {"category": 144, "azimuth": 252.37018273111011},
                {"category": 145, "azimuth": 253.32980157392765},
                {"category": 146, "azimuth": 73.329801573927668},
                {"category": 147, "azimuth": 146.92587780562783},
                {"category": 148, "azimuth": 64.814046737607001},
                {"category": 149, "azimuth": 352.46322241250488},
                {"category": 150, "azimuth": 121.99900614608642},
                {"category": 151, "azimuth": 66.10317533072967},
                {"category": 152, "azimuth": 146.90891610775887},
                {"category": 153, "azimuth": 146.92245048943013},
                {"category": 154, "azimuth": 188.03533900279254},
                {"category": 155, "azimuth": 352.41932073195329},
                {"category": 156, "azimuth": 146.92400686501168},
                {"category": 157, "azimuth": 356.0658872995154},
                {"category": 158, "azimuth": 313.29663635382462},
                {"category": 159, "azimuth": 116.82444094566304},
                {"category": 160, "azimuth": 71.760204782539262},
                {"category": 161, "azimuth": 5.934260972712484},
                {"category": 162, "azimuth": 10.439998339875199},
                {"category": 163, "azimuth": 180.00239320741912},
                {"category": 164, "azimuth": 31.96165313646441},
                {"category": 165, "azimuth": 358.06369156276082},
                {"category": 166, "azimuth": 330.93232770659108},
                {"category": 167, "azimuth": 62.632878815615975},
                {"category": 168, "azimuth": 218.86199366592041},
                {"category": 169, "azimuth": 87.899910478923545},
                {"category": 170, "azimuth": 228.60184043152023},
                {"category": 171, "azimuth": 107.00137050344271},
                {"category": 172, "azimuth": 271.82769224480518},
                {"category": 173, "azimuth": 97.570625792435635},
                {"category": 174, "azimuth": 277.57062579243564},
                {"category": 175, "azimuth": 358.75349911888532},
                {"category": 176, "azimuth": 283.27064386924201},
                {"category": 177, "azimuth": 283.26332874688984},
                {"category": 178, "azimuth": 103.23655928534728},
                {"category": 179, "azimuth": 283.23655928534725},
                {"category": 180, "azimuth": 359.12105420640404},
                {"category": 181, "azimuth": 240.62213902893285},
                {"category": 182, "azimuth": 339.760521968099},
                {"category": 183, "azimuth": 287.80891706256085},
                {"category": 184, "azimuth": 95.188700326066765},
                {"category": 185, "azimuth": 286.87143238454019},
                {"category": 186, "azimuth": 283.23655928534725},
                {"category": 187, "azimuth": 26.255608254972422},
                {"category": 188, "azimuth": 0.54007349581838693},
                {"category": 189, "azimuth": 129.93336570091142},
                {"category": 190, "azimuth": 80.761475377096104},
                {"category": 191, "azimuth": 7.0980870127748759},
                {"category": 192, "azimuth": 137.14550342020519},
                {"category": 193, "azimuth": 316.0763217670202},
                {"category": 194, "azimuth": 284.51035937149754},
                {"category": 195, "azimuth": 230.55487089661011},
                {"category": 196, "azimuth": 275.01624074126329},
                {"category": 197, "azimuth": 14.917773896569907},
                {"category": 198, "azimuth": 99.80854334463335},
                {"category": 199, "azimuth": 279.80854334463339},
                {"category": 200, "azimuth": 13.847264568623082},
                {"category": 201, "azimuth": 84.797906261355919},
                {"category": 202, "azimuth": 101.32660082062003},
                {"category": 203, "azimuth": 342.38311053807467},
                {"category": 204, "azimuth": 26.017234832427484},
                {"category": 205, "azimuth": 347.77388067471207},
                {"category": 206, "azimuth": 53.320800311397001},
                {"category": 207, "azimuth": 130.33115152379023},
                {"category": 208, "azimuth": 227.59915070728636},
                {"category": 209, "azimuth": 340.68489265475932},
                {"category": 210, "azimuth": 130.41379159472891},
                {"category": 211, "azimuth": 90.708484054160991},
                {"category": 212, "azimuth": 26.238562957026417},
                {"category": 213, "azimuth": 83.348984289773},
                {"category": 214, "azimuth": 120.96000656356671},
                {"category": 215, "azimuth": 26.303029622191428},
                {"category": 216, "azimuth": 119.16042312898428},
                {"category": 217, "azimuth": 26.266638353209789},
                {"category": 218, "azimuth": 119.13591205949491},
                {"category": 219, "azimuth": 8.9807805665023892},
                {"category": 220, "azimuth": 299.22112527016725},
                {"category": 221, "azimuth": 274.97929285088321},
                {"category": 222, "azimuth": 293.30552983803875},
                {"category": 223, "azimuth": 50.529360553156934},
                {"category": 224, "azimuth": 28.288806542666705},
                {"category": 225, "azimuth": 29.475889004669217},
                {"category": 226, "azimuth": 186.21608555843554},
                {"category": 227, "azimuth": 263.72662506797485},
                {"category": 228, "azimuth": 353.86996004708362},
                {"category": 229, "azimuth": 221.89496145068242},
                {"category": 230, "azimuth": 300.23462700682666},
                {"category": 231, "azimuth": 210.1283559054518},
                {"category": 232, "azimuth": 166.71389297816756},
                {"category": 233, "azimuth": 340.94614014623096},
                {"category": 234, "azimuth": 242.4048447963398},
                {"category": 235, "azimuth": 290.40921592894193},
                {"category": 236, "azimuth": 325.83806176443005},
                {"category": 237, "azimuth": 329.3482967562191},
                {"category": 238, "azimuth": 66.539423828548493},
                {"category": 239, "azimuth": 107.57807894679399},
                {"category": 240, "azimuth": 64.107736076775993},
                {"category": 241, "azimuth": 330.11037178134069},
                {"category": 242, "azimuth": 329.69513562309845},
                {"category": 243, "azimuth": 330.37324514210496},
                {"category": 244, "azimuth": 335.6347058839691},
                {"category": 245, "azimuth": 193.71922106472542},
                {"category": 246, "azimuth": 332.79598892387628},
                {"category": 247, "azimuth": 157.08408515316532},
                {"category": 248, "azimuth": 63.457816824012319},
                {"category": 249, "azimuth": 177.52233730643226},
                {"category": 250, "azimuth": 59.568151870461406},
                {"category": 251, "azimuth": 27.831552146105299},
                {"category": 252, "azimuth": 338.48582048431865},
                {"category": 253, "azimuth": 340.61548432330665},
                {"category": 254, "azimuth": 114.59116965765492},
                {"category": 255, "azimuth": 291.85705008628406},
                {"category": 256, "azimuth": 291.85705008628406},
                {"category": 257, "azimuth": 271.66939756491354},
                {"category": 258, "azimuth": 277.47363576277053},
                {"category": 259, "azimuth": 276.15375580947915},
                {"category": 260, "azimuth": 271.0851059900013},
                {"category": 261, "azimuth": 209.4480262817176},
                {"category": 262, "azimuth": 255.74691062907962},
                {"category": 263, "azimuth": 279.60991826213728},
                {"category": 264, "azimuth": 214.75054047033422},
                {"category": 265, "azimuth": 128.98585860993555},
                {"category": 266, "azimuth": 177.67723983042498},
                {"category": 267, "azimuth": 357.67723983042498},
                {"category": 268, "azimuth": 287.39436510900083},
                {"category": 269, "azimuth": 270.95031677035769},
                {"category": 270, "azimuth": 258.5091202120858},
                {"category": 271, "azimuth": 266.16519218249607},
                {"category": 272, "azimuth": 58.691016107058282},
                {"category": 273, "azimuth": 266.02805786095973},
                {"category": 274, "azimuth": 200.40598899462324},
                {"category": 275, "azimuth": 177.67381191269837},
                {"category": 276, "azimuth": 124.62120582227678},
                {"category": 277, "azimuth": 288.46413399604461},
                {"category": 278, "azimuth": 56.40469698463076},
                {"category": 279, "azimuth": 178.10962691119479},
                {"category": 280, "azimuth": 49.811680729014796},
                {"category": 281, "azimuth": 348.95931792778282},
                {"category": 282, "azimuth": 163.6187034863166},
                {"category": 283, "azimuth": 163.6187034863166},
                {"category": 284, "azimuth": 4.8479492265223563},
                {"category": 285, "azimuth": 353.33173991064984},
                {"category": 286, "azimuth": 319.60103043106852},
                {"category": 287, "azimuth": 273.00207139416119},
                {"category": 288, "azimuth": 263.54927704344902},
                {"category": 289, "azimuth": 262.8947012568529},
                {"category": 290, "azimuth": 56.3087966183151},
                {"category": 291, "azimuth": 285.72596452412409},
                {"category": 292, "azimuth": 252.84116257625647},
                {"category": 293, "azimuth": 108.94074317342287},
                {"category": 294, "azimuth": 101.78152283130223},
                {"category": 295, "azimuth": 57.987992389568568},
                {"category": 296, "azimuth": 102.26073511761838},
                {"category": 297, "azimuth": 57.762537066655305},
                {"category": 298, "azimuth": 344.04183083028585},
                {"category": 299, "azimuth": 57.620142972115005},
                {"category": 300, "azimuth": 190.36976076392469},
                {"category": 301, "azimuth": 324.7328108312401},
                {"category": 302, "azimuth": 181.7236703201734},
                {"category": 303, "azimuth": 162.95611004023317},
                {"category": 304, "azimuth": 61.394408615061757},
                {"category": 305, "azimuth": 41.883011399785651},
                {"category": 306, "azimuth": 231.33948353353961},
                {"category": 307, "azimuth": 314.73206129802907},
                {"category": 308, "azimuth": 38.827702701346652},
                {"category": 309, "azimuth": 156.66390188004345},
                {"category": 310, "azimuth": 27.215698535910089},
                {"category": 311, "azimuth": 156.67013716594715},
                {"category": 312, "azimuth": 39.362106848004601},
                {"category": 313, "azimuth": 207.04739286476513},
                {"category": 314, "azimuth": 56.048765057348866},
                {"category": 315, "azimuth": 47.363246652791418},
                {"category": 316, "azimuth": 124.27471403962242},
                {"category": 317, "azimuth": 163.42443076000791},
                {"category": 318, "azimuth": 289.77932905389503},
                {"category": 319, "azimuth": 211.16343129385362},
                {"category": 320, "azimuth": 237.93433041339256},
                {"category": 321, "azimuth": 111.02294950047843},
                {"category": 322, "azimuth": 171.21495736726632},
                {"category": 323, "azimuth": 238.52018067940682},
                {"category": 324, "azimuth": 118.7946439045551},
                {"category": 325, "azimuth": 170.15439369483761},
                {"category": 326, "azimuth": 215.611479049228},
                {"category": 327, "azimuth": 166.40052735524768},
                {"category": 328, "azimuth": 239.47810339164985},
                {"category": 329, "azimuth": 242.45117076319869},
                {"category": 330, "azimuth": 177.66211893241586},
                {"category": 331, "azimuth": 186.00655627396287},
                {"category": 332, "azimuth": 201.56676158689118},
                {"category": 333, "azimuth": 191.93346775259317},
                {"category": 334, "azimuth": 145.84708840340721},
                {"category": 335, "azimuth": 158.97454435007015},
                {"category": 336, "azimuth": 225.83401828557356},
                {"category": 337, "azimuth": 235.10650381881254},
                {"category": 338, "azimuth": 110.35337224083261},
                {"category": 339, "azimuth": 145.46342578078099},
                {"category": 340, "azimuth": 86.264653066354697},
                {"category": 341, "azimuth": 77.057804397379002},
                {"category": 342, "azimuth": 96.85348975313731},
                {"category": 343, "azimuth": 349.35608740948032},
                {"category": 344, "azimuth": 87.432928901842544},
                {"category": 345, "azimuth": 91.405213433518895},
                {"category": 346, "azimuth": 79.424297419192371},
                {"category": 347, "azimuth": 37.561157032737434},
                {"category": 348, "azimuth": 103.71611278770719},
                {"category": 349, "azimuth": 254.39166903053467},
                {"category": 350, "azimuth": 238.92549116932506},
                {"category": 351, "azimuth": 353.88501281080801},
                {"category": 352, "azimuth": 197.04610062808683},
                {"category": 353, "azimuth": 344.49674091143578},
                {"category": 354, "azimuth": 100.04388278372213},
                {"category": 355, "azimuth": 235.63026574838457},
            ],
        }
        module = SimpleModule(
            "v.to.db",
            "roadsmajor@PERMANENT",
            flags="p",
            option="azimuth",
            type="line",
            units="degrees",
            format="json",
        )
        self._assert_json_equal(module, reference_deg)

    def test_json_sinuous(self):
        reference = {
            "units": {},
            "totals": {},
            "records": [
                {"category": 1, "sinuous": 1.0003414320997843},
                {"category": 2, "sinuous": 1.0176449991654128},
                {"category": 3, "sinuous": 1.0002427144064336},
                {"category": 4, "sinuous": 1.0172749216604193},
                {"category": 5, "sinuous": 1.049283149801576},
                {"category": 6, "sinuous": 1.045248721258855},
                {"category": 7, "sinuous": 1.0199254015587185},
                {"category": 8, "sinuous": 1.3081558062240128},
                {"category": 9, "sinuous": 1.026683249842919},
                {"category": 10, "sinuous": 1.0408137145875025},
                {"category": 11, "sinuous": 1.0137641054279147},
                {"category": 12, "sinuous": 1.0296847800653639},
                {"category": 13, "sinuous": 1.022399447677268},
                {"category": 14, "sinuous": 1.0095758126232299},
                {"category": 15, "sinuous": 1.000001372226867},
                {"category": 16, "sinuous": 1.0022552627600008},
                {"category": 17, "sinuous": 1.0393353412289232},
                {"category": 18, "sinuous": 1.0011052600711821},
                {"category": 19, "sinuous": 1.037076304639921},
                {"category": 20, "sinuous": 1.0073680096997011},
                {"category": 21, "sinuous": 1.0426730046188117},
                {"category": 22, "sinuous": 1.0129073941779174},
                {"category": 23, "sinuous": 1.0010623961240077},
                {"category": 24, "sinuous": 1.0473767485119543},
                {"category": 25, "sinuous": 1.019113917782827},
                {"category": 26, "sinuous": 1.0025890027188795},
                {"category": 27, "sinuous": 1.0295814732560054},
                {"category": 28, "sinuous": 1.0079557277922209},
                {"category": 29, "sinuous": 1.0001299865186111},
                {"category": 30, "sinuous": 1.0036231984185473},
                {"category": 31, "sinuous": 1.0001262640878104},
                {"category": 32, "sinuous": 1.0068387880191103},
                {"category": 33, "sinuous": 1.0008938104176925},
                {"category": 34, "sinuous": 1.0378064902277309},
                {"category": 35, "sinuous": 1.0882935819773731},
                {"category": 36, "sinuous": 1.0015975153373733},
                {"category": 37, "sinuous": 1.0001414856708983},
                {"category": 38, "sinuous": 1.0300495723910386},
                {"category": 39, "sinuous": 1.0064800311110842},
                {"category": 40, "sinuous": 1.0000244086228074},
                {"category": 41, "sinuous": 1.0000000340118238},
                {"category": 42, "sinuous": 1.0002319307529177},
                {"category": 43, "sinuous": 1.0000010519349034},
                {"category": 44, "sinuous": 1.000389732828896},
                {"category": 45, "sinuous": 1.0064509309084844},
                {"category": 46, "sinuous": 1.0000006703786373},
                {"category": 47, "sinuous": 1.0128352762472352},
                {"category": 48, "sinuous": 1},
                {"category": 49, "sinuous": 1},
                {"category": 50, "sinuous": 1.0025113575929119},
                {"category": 51, "sinuous": 1.0106782232205278},
                {"category": 52, "sinuous": 1.001000965350467},
                {"category": 53, "sinuous": 1.0146632476118098},
                {"category": 54, "sinuous": 1.0449483617751267},
                {"category": 55, "sinuous": 1.0054268967880473},
                {"category": 56, "sinuous": 1.0026253416984983},
                {"category": 57, "sinuous": 1.0081732607687064},
                {"category": 58, "sinuous": 1.0000007067779717},
                {"category": 59, "sinuous": 1},
                {"category": 60, "sinuous": 1},
                {"category": 61, "sinuous": 1.0440227360435339},
                {"category": 62, "sinuous": 1.0197162701686162},
                {"category": 63, "sinuous": 1},
                {"category": 64, "sinuous": 2.8364193495341992},
                {"category": 65, "sinuous": 1.0136759762419687},
                {"category": 66, "sinuous": 1.0000347797079683},
                {"category": 67, "sinuous": 1.0000016983185038},
                {"category": 68, "sinuous": 1.1256193883592067},
                {"category": 69, "sinuous": 1.0050479171062636},
                {"category": 70, "sinuous": 1.0000007055293487},
                {"category": 71, "sinuous": 1.7055829061553012},
                {"category": 72, "sinuous": 1.0000000024652926},
                {"category": 73, "sinuous": 1.000156299513749},
                {"category": 74, "sinuous": 1.0000011927702956},
                {"category": 75, "sinuous": 1.0002109480122563},
                {"category": 76, "sinuous": 3.2624446635269124},
                {"category": 77, "sinuous": 1.0002193404710145},
                {"category": 78, "sinuous": 1.032420178351712},
                {"category": 79, "sinuous": 1.0025511856117599},
                {"category": 80, "sinuous": 1.000227894548257},
                {"category": 81, "sinuous": 1.0000015721755273},
                {"category": 82, "sinuous": 1.0000014042930108},
                {"category": 83, "sinuous": 1.0011383273283412},
                {"category": 84, "sinuous": 6.5671204129175749},
                {"category": 85, "sinuous": 1.0000005720364493},
                {"category": 86, "sinuous": 1.0307382101837275},
                {"category": 87, "sinuous": 1.0000282660482764},
                {"category": 88, "sinuous": 1.003910311317149},
                {"category": 89, "sinuous": 1.0000011445246391},
                {"category": 90, "sinuous": 1.0000004853088409},
                {"category": 91, "sinuous": 1.014456180543291},
                {"category": 92, "sinuous": 2.9095482983059968},
                {"category": 93, "sinuous": 1.0009629782022398},
                {"category": 94, "sinuous": 1.1202925894938389},
                {"category": 95, "sinuous": 1.0014697286203897},
                {"category": 96, "sinuous": 1.0066017292228973},
                {"category": 97, "sinuous": 1.0022366460969527},
                {"category": 98, "sinuous": 1.0000012680066244},
                {"category": 99, "sinuous": 1.0774069713598595},
                {"category": 100, "sinuous": 1.0001243784791005},
                {"category": 101, "sinuous": 1.03024948004905},
                {"category": 102, "sinuous": 1.0000004538521745},
                {"category": 103, "sinuous": 1.0005587713493909},
                {"category": 104, "sinuous": 1.0000000627888543},
                {"category": 105, "sinuous": 1.020602594676971},
                {"category": 106, "sinuous": 1.0004934954627138},
                {"category": 107, "sinuous": 1.0259596174263492},
                {"category": 108, "sinuous": 1.0004771153855281},
                {"category": 109, "sinuous": 1},
                {"category": 110, "sinuous": 1.0152880800505404},
                {"category": 111, "sinuous": 1.0029337902565862},
                {"category": 112, "sinuous": 1.0000439822052516},
                {"category": 113, "sinuous": 1.0098394456410418},
                {"category": 114, "sinuous": 1.0000089866788164},
                {"category": 115, "sinuous": 1.0005703292237449},
                {"category": 116, "sinuous": 1.0092048182443387},
                {"category": 117, "sinuous": 1},
                {"category": 118, "sinuous": 1.0154495345299326},
                {"category": 119, "sinuous": 1.0003385451518285},
                {"category": 120, "sinuous": 1.0000006857619086},
                {"category": 121, "sinuous": 1.0000607755423963},
                {"category": 122, "sinuous": 1.02238550649198},
                {"category": 123, "sinuous": 1.0000004139921519},
                {"category": 124, "sinuous": 1.0000020312331215},
                {"category": 125, "sinuous": 1.0000007007887313},
                {"category": 126, "sinuous": 1.0204465217129217},
                {"category": 127, "sinuous": 1.0009426412154621},
                {"category": 128, "sinuous": 1.0043252039529615},
                {"category": 129, "sinuous": 1.0000008066726833},
                {"category": 130, "sinuous": 1.0237533783557315},
                {"category": 131, "sinuous": 1.0000225102169955},
                {"category": 132, "sinuous": 1.0000190616059472},
                {"category": 133, "sinuous": 1.0000009417792319},
                {"category": 134, "sinuous": 20.58653166054351},
                {"category": 135, "sinuous": 1.0000004481378768},
                {"category": 136, "sinuous": 1.0271165536793037},
                {"category": 137, "sinuous": 1.0000005578977886},
                {"category": 138, "sinuous": 1.0470528257570726},
                {"category": 139, "sinuous": 1.0531205774120562},
                {"category": 140, "sinuous": 1.0117439365485763},
                {"category": 141, "sinuous": 1.0000011512039499},
                {"category": 142, "sinuous": 1.002593214346716},
                {"category": 143, "sinuous": 1.0000008534516382},
                {"category": 144, "sinuous": 1.0059407738579451},
                {"category": 145, "sinuous": 1.0000001461905821},
                {"category": 146, "sinuous": 9.4506634407543775},
                {"category": 147, "sinuous": 1.0000011539419009},
                {"category": 148, "sinuous": 1.0000010301404409},
                {"category": 149, "sinuous": 1.0015363369144459},
                {"category": 150, "sinuous": 1.000000338659321},
                {"category": 151, "sinuous": 1.0000010605671303},
                {"category": 152, "sinuous": 1.0000004994894649},
                {"category": 153, "sinuous": 1.0000017591867238},
                {"category": 154, "sinuous": 2.1314719364917374},
                {"category": 155, "sinuous": 1.0163491124943627},
                {"category": 156, "sinuous": 1.0000009975105093},
                {"category": 157, "sinuous": 1.001203368855488},
                {"category": 158, "sinuous": 1.0000011487744491},
                {"category": 159, "sinuous": 1.0116770130045727},
                {"category": 160, "sinuous": 1.2047729233824027},
                {"category": 161, "sinuous": 1.0000367274363264},
                {"category": 162, "sinuous": 1.0000008024798634},
                {"category": 163, "sinuous": 1.038948683914303},
                {"category": 164, "sinuous": 1.0331549590459059},
                {"category": 165, "sinuous": 1.0000178587779676},
                {"category": 166, "sinuous": 1.000068454365209},
                {"category": 167, "sinuous": 1.0003242054403361},
                {"category": 168, "sinuous": 1.0124385225546786},
                {"category": 169, "sinuous": 1.0049728809654748},
                {"category": 170, "sinuous": 1.0925563463258312},
                {"category": 171, "sinuous": 1.0356799862967603},
                {"category": 172, "sinuous": 1.0006797758483481},
                {"category": 173, "sinuous": 2.3966747800996098},
                {"category": 174, "sinuous": 1.0000727235412186},
                {"category": 175, "sinuous": 1.0000258482418545},
                {"category": 176, "sinuous": 1},
                {"category": 177, "sinuous": 1},
                {"category": 178, "sinuous": 4.020700828530015},
                {"category": 179, "sinuous": 1.0000008521382711},
                {"category": 180, "sinuous": 1.0000006311855263},
                {"category": 181, "sinuous": 1.0364815877818128},
                {"category": 182, "sinuous": 1.0410451248736321},
                {"category": 183, "sinuous": 1.00000099550858},
                {"category": 184, "sinuous": 1.008459924628228},
                {"category": 185, "sinuous": 1.8906557745906343},
                {"category": 186, "sinuous": 4.1545118974624797},
                {"category": 187, "sinuous": 1.3508716804072907},
                {"category": 188, "sinuous": 1.0000422424382618},
                {"category": 189, "sinuous": 1.0800780549336708},
                {"category": 190, "sinuous": 1.0904231362164276},
                {"category": 191, "sinuous": 1.0026147790815114},
                {"category": 192, "sinuous": 1.0047320328410045},
                {"category": 193, "sinuous": 1.0863008539161043},
                {"category": 194, "sinuous": 1.0200337553488408},
                {"category": 195, "sinuous": 1.0558929748954857},
                {"category": 196, "sinuous": 1.0011537064091003},
                {"category": 197, "sinuous": 1.0015478386630288},
                {"category": 198, "sinuous": 3.6411515694429788},
                {"category": 199, "sinuous": 1.0009415641632167},
                {"category": 200, "sinuous": 1.0025493366985143},
                {"category": 201, "sinuous": 1.0206961289745298},
                {"category": 202, "sinuous": 2.9357134430336318},
                {"category": 203, "sinuous": 1.0193455501982236},
                {"category": 204, "sinuous": 1.0000011571998451},
                {"category": 205, "sinuous": 1.0572059961784517},
                {"category": 206, "sinuous": 1.0000472301709964},
                {"category": 207, "sinuous": 1.0000007904652826},
                {"category": 208, "sinuous": 1.0173819001613302},
                {"category": 209, "sinuous": 1.0000019752107381},
                {"category": 210, "sinuous": 1.0000017239051964},
                {"category": 211, "sinuous": 1.0913357600900013},
                {"category": 212, "sinuous": 1.0000031987670839},
                {"category": 213, "sinuous": 2.4286615538870309},
                {"category": 214, "sinuous": 1.0000007326276494},
                {"category": 215, "sinuous": 1.0000000083021634},
                {"category": 216, "sinuous": 1.0000005159822296},
                {"category": 217, "sinuous": 1},
                {"category": 218, "sinuous": 1.0000000371391171},
                {"category": 219, "sinuous": 1.009165291611906},
                {"category": 220, "sinuous": 1.0000015297390012},
                {"category": 221, "sinuous": 1.077082383713948},
                {"category": 222, "sinuous": 3.1075635395346701},
                {"category": 223, "sinuous": 1.014117566025974},
                {"category": 224, "sinuous": 1.0004173018230662},
                {"category": 225, "sinuous": 1.0000005566437111},
                {"category": 226, "sinuous": 1.0070906559988313},
                {"category": 227, "sinuous": 1.0423361414269392},
                {"category": 228, "sinuous": 1.0057046303389867},
                {"category": 229, "sinuous": 1.0141036789246773},
                {"category": 230, "sinuous": 1.0710371280663418},
                {"category": 231, "sinuous": 1.0279219505861106},
                {"category": 232, "sinuous": 1.0695444649073167},
                {"category": 233, "sinuous": 1.1095103041801577},
                {"category": 234, "sinuous": 1.0051720825713819},
                {"category": 235, "sinuous": 1.0431100923911307},
                {"category": 236, "sinuous": 1.0009483122061842},
                {"category": 237, "sinuous": 1.0000000961928763},
                {"category": 238, "sinuous": 1.000107528208716},
                {"category": 239, "sinuous": 5.8588618793368363},
                {"category": 240, "sinuous": 1.0000000462362681},
                {"category": 241, "sinuous": 1.0003651831709646},
                {"category": 242, "sinuous": 3.3471604215141784},
                {"category": 243, "sinuous": 1.0019358141649772},
                {"category": 244, "sinuous": 1.0000015320659112},
                {"category": 245, "sinuous": 5.7192191543523085},
                {"category": 246, "sinuous": 3.0606459098893675},
                {"category": 247, "sinuous": 1.054627619395347},
                {"category": 248, "sinuous": 1.0000622907944063},
                {"category": 249, "sinuous": 1.1075080963762451},
                {"category": 250, "sinuous": 1.0001138666403504},
                {"category": 251, "sinuous": 1.0346603319961405},
                {"category": 252, "sinuous": 1.0011204496329646},
                {"category": 253, "sinuous": 1},
                {"category": 254, "sinuous": 1.0629063069880709},
                {"category": 255, "sinuous": 1.0290980668243785},
                {"category": 256, "sinuous": 1.0012327003644239},
                {"category": 257, "sinuous": 1.0001262022490232},
                {"category": 258, "sinuous": 1.0091615063690897},
                {"category": 259, "sinuous": 1.0727734447935322},
                {"category": 260, "sinuous": 1.0000168226189945},
                {"category": 261, "sinuous": 1.1418699330773918},
                {"category": 262, "sinuous": 1.0000005297112839},
                {"category": 263, "sinuous": 1.0076699769692419},
                {"category": 264, "sinuous": 1.0157685918632993},
                {"category": 265, "sinuous": 1.0166687055143095},
                {"category": 266, "sinuous": 1.0000007622514224},
                {"category": 267, "sinuous": 2.1165700994326362},
                {"category": 268, "sinuous": 1.0083709576387501},
                {"category": 269, "sinuous": 1.0000000181040922},
                {"category": 270, "sinuous": 1.0023363647830672},
                {"category": 271, "sinuous": 1.0566274965394409},
                {"category": 272, "sinuous": 1.0005907635114175},
                {"category": 273, "sinuous": 1.0316600451152369},
                {"category": 274, "sinuous": 1.7690012657172149},
                {"category": 275, "sinuous": 1.0000008052768816},
                {"category": 276, "sinuous": 1.000001146448467},
                {"category": 277, "sinuous": 1.057751675851093},
                {"category": 278, "sinuous": 1.028163081931144},
                {"category": 279, "sinuous": 1.0000009979352822},
                {"category": 280, "sinuous": 1.1203822660318314},
                {"category": 281, "sinuous": 1.0044045329190903},
                {"category": 282, "sinuous": 1.0066956395827185},
                {"category": 283, "sinuous": 1.0611155757276394},
                {"category": 284, "sinuous": 1.0182792831897538},
                {"category": 285, "sinuous": 1.0009344997586904},
                {"category": 286, "sinuous": 1.0000005860686043},
                {"category": 287, "sinuous": 1.0173690652154948},
                {"category": 288, "sinuous": 1.0001065635522428},
                {"category": 289, "sinuous": 1.0000507653074835},
                {"category": 290, "sinuous": 1.0001009861368446},
                {"category": 291, "sinuous": 1.0015797782143501},
                {"category": 292, "sinuous": 1.0027519518573029},
                {"category": 293, "sinuous": 1.0000010790518772},
                {"category": 294, "sinuous": 1.0006017029403373},
                {"category": 295, "sinuous": 1.0000026509844557},
                {"category": 296, "sinuous": 1.0001268638752114},
                {"category": 297, "sinuous": 1.000000311869808},
                {"category": 298, "sinuous": 3.4065143333776753},
                {"category": 299, "sinuous": 1.0000134171948452},
                {"category": 300, "sinuous": 1.1955759905143186},
                {"category": 301, "sinuous": 1.0091625988970496},
                {"category": 302, "sinuous": 1.0101154289043888},
                {"category": 303, "sinuous": 1.0019715580236945},
                {"category": 304, "sinuous": 1.0102063974946556},
                {"category": 305, "sinuous": 1.000041966815717},
                {"category": 306, "sinuous": 1.0033111417246503},
                {"category": 307, "sinuous": 1.0490534028168135},
                {"category": 308, "sinuous": 1.0002705925317035},
                {"category": 309, "sinuous": 1.0000006778549702},
                {"category": 310, "sinuous": 1.0125586137376883},
                {"category": 311, "sinuous": 1.0000009122017552},
                {"category": 312, "sinuous": 1.0015912221414836},
                {"category": 313, "sinuous": 1.0282676103497261},
                {"category": 314, "sinuous": 1.0009622998426706},
                {"category": 315, "sinuous": 1},
                {"category": 316, "sinuous": 1.0075457872111493},
                {"category": 317, "sinuous": 1.0525930249315485},
                {"category": 318, "sinuous": 1.0050053507812666},
                {"category": 319, "sinuous": 1.0352310656865686},
                {"category": 320, "sinuous": 1.0009893391762794},
                {"category": 321, "sinuous": 1.0016067416375256},
                {"category": 322, "sinuous": 1.0187264305737924},
                {"category": 323, "sinuous": 1.0000194643179903},
                {"category": 324, "sinuous": 1.0000048912002526},
                {"category": 325, "sinuous": 1.0322839462790272},
                {"category": 326, "sinuous": 1.0276463102386912},
                {"category": 327, "sinuous": 1.0306220755505615},
                {"category": 328, "sinuous": 1.0011410233786282},
                {"category": 329, "sinuous": 1.0054492864871649},
                {"category": 330, "sinuous": 1.018747200948001},
                {"category": 331, "sinuous": 1.0683320177430757},
                {"category": 332, "sinuous": 1.009518287691785},
                {"category": 333, "sinuous": 1.0107978321614883},
                {"category": 334, "sinuous": 1.0013387905477116},
                {"category": 335, "sinuous": 1.0160205099039565},
                {"category": 336, "sinuous": 1.0009495301588032},
                {"category": 337, "sinuous": 1.0094148319806282},
                {"category": 338, "sinuous": 1.0000010458129649},
                {"category": 339, "sinuous": 1.0192114415678681},
                {"category": 340, "sinuous": 1.000009344662975},
                {"category": 341, "sinuous": 1.0110239046772556},
                {"category": 342, "sinuous": 1.020678763434637},
                {"category": 343, "sinuous": 1.0400994863405235},
                {"category": 344, "sinuous": 1.0007992824191538},
                {"category": 345, "sinuous": 1.0370729518236121},
                {"category": 346, "sinuous": 1.001515647567724},
                {"category": 347, "sinuous": 1.2264110346486641},
                {"category": 348, "sinuous": 1.0363016389599002},
                {"category": 349, "sinuous": 1.0166058536067142},
                {"category": 350, "sinuous": 1.0010601901034746},
                {"category": 351, "sinuous": 1.0018402555229273},
                {"category": 352, "sinuous": 1.0317799263379368},
                {"category": 353, "sinuous": 1.069456444203726},
                {"category": 354, "sinuous": 1.0874373808675006},
                {"category": 355, "sinuous": 1.0515676406887557},
            ],
        }
        module = SimpleModule(
            "v.to.db",
            "roadsmajor@PERMANENT",
            flags="p",
            option="sinuous",
            format="json",
        )
        self._assert_json_equal(module, reference)


if __name__ == "__main__":
    test()
