## DESCRIPTION

*r3.mapcalc* performs arithmetic on raster map layers. New raster map
layers can be created which are arithmetic expressions involving
existing raster map layers, integer or floating point constants, and
functions.

### Program use

*r3.mapcalc* expression have the form:

**result =** *expression*

where *result* is the name of a raster map layer to contain the result
of the calculation and **expression** is any legal arithmetic expression
involving existing raster map layers (except *result* itself), integer
or floating point constants, and functions known to the calculator.
Parentheses are allowed in the expression and may be nested to any
depth. *result* will be created in the user's current mapset.

As **expression=** is the first option, it is the default. This means
that passing an expression on the command line is possible as long as
the expression is quoted and a space is included before the first *=*
sign. Example ('foo' is the resulting map):

```sh
r3.mapcalc "foo = 1"
```

or:

```sh
r3.mapcalc 'foo = 1'
```

An unquoted expression (i.e. split over multiple arguments) won't work,
nor will omitting the space before the = sign:

```sh
r3.mapcalc 'foo=1'
Sorry, <foo> is not a valid parameter
```

If no options are given, it manufactures "file=-" (which reads from
stdin), so you can continue to use e.g.:

```sh
r3.mapcalc < file
```

or:

```sh
r3.mapcalc <<EOF
foo = 1
EOF
```

But unless you need compatibility with previous versions, use file=
explicitly, e.g.:

```sh
r3.mapcalc file=file
```

or:

```sh
r3.mapcalc file=- <<EOF
foo = 1
EOF
```

The formula entered to *r3.mapcalc* by the user is recorded both in the
*result* map title (which appears in the category file for *result*) and
in the history file for *result*.

Some characters have special meaning to the command shell. If the user
is entering input to *r3.mapcalc* on the command line, expressions
should be enclosed within single quotes. See NOTES, below.

### Operators and order of precedence

The following operators are supported:

| Operator | Meaning                    | Type        | Precedence |
|----------|----------------------------|-------------|------------|
| -        | negation                   | Arithmetic  | 12         |
| ~        | one's complement           | Bitwise     | 12         |
| !        | not                        | Logical     | 12         |
| ^        | exponentiation             | Arithmetic  | 11         |
| %        | modulus                    | Arithmetic  | 10         |
| /        | division                   | Arithmetic  | 10         |
| *        | multiplication             | Arithmetic  | 10         |
| +        | addition                   | Arithmetic  | 9          |
| -        | subtraction                | Arithmetic  | 9          |
| <<       | left shift                 | Bitwise     | 8          |
| >>       | right shift                | Bitwise     | 8          |
| >>>      | right shift (unsigned)     | Bitwise     | 8          |
| >        | greater than               | Logical     | 7          |
| >=       | greater than or equal      | Logical     | 7          |
| <        | less than                  | Logical     | 7          |
| <=       | less than or equal         | Logical     | 7          |
| ==       | equal                      | Logical     | 6          |
| !=       | not equal                  | Logical     | 6          |
| &        | bitwise and                | Bitwise     | 5          |
| \|       | bitwise or                 | Bitwise     | 4          |
| &&       | logical and                | Logical     | 3          |
| &&&      | logical and\[1\]           | Logical     | 3          |
| \|\|     | logical or                 | Logical     | 2          |
| \|\|\|   | logical or\[1\]            | Logical     | 2          |
| ?:       | conditional                | Logical     | 1          |

(modulus is the remainder upon division)

\[1\] The &&& and \|\|\| operators handle null values differently to
other operators. See the section entitled **NULL support** below for
more details.

The operators are applied from left to right, with those of higher
precedence applied before those with lower precedence. Division by 0 and
modulus by 0 are acceptable and give a NULL result. The logical
operators give a 1 result if the comparison is true, 0 otherwise.

### 3D Grid names

Anything in the expression which is not a number, operator, or function
name is taken to be a 3D grid name. Examples:

```sh
volume
x3
3d.his
```

Most GRASS raster map layers and 3D grids meet this naming convention.
However, if a 3D grid has a name which conflicts with the above rule, it
should be quoted. For example, the expression

```sh
x = a-b
```

would be interpreted as: x equals a minus b, whereas

```sh
x = "a-b"
```

would be interpreted as: x equals the 3D grid named *a-b*

Also

```sh
x = 3107
```

would create *x* filled with the number 3107, while

```sh
x = "3107"
```

would copy the 3D grid *3107* to the 3D grid *x*.

Quotes are not required unless the 3D grid names look like numbers or
contain operators, OR unless the program is run non-interactively.
Examples given here assume the program is run interactively. See NOTES,
below.

*r3.mapcalc* will look for the 3D grids according to the user's current
mapset search path. It is possible to override the search path and
specify the mapset from which to select the 3D grid. This is done by
specifying the 3D grid name in the form:

```sh
name@mapset
```

For example, the following is a legal expression:

```sh
result = x@PERMANENT / y@SOILS
```

The mapset specified does not have to be in the mapset search path.
(This method of overriding the mapset search path is common to all GRASS
commands, not just *r3.mapcalc*.)

### The neighborhood modifier

3D grids are data base files stored in voxel format, i.e.,
three-dimensional matrices of float/double values. In *r3.mapcalc*, 3D
grids may be followed by a *neighborhood* modifier that specifies a
relative offset from the current cell being evaluated. The format is
*map\[r,c,d\]*, where *r* is the row offset, *c* is the column offset
and *d* is the depth offset. For example, *map\[1,2,3\]* refers to the
cell one row below, two columns to the right and 3 levels below of the
current cell, *map\[-3,-2,-1\]* refers to the cell three rows above, two
columns to the left and one level below of the current cell, and
*map\[0,1,0\]* refers to the cell one column to the right of the current
cell. This syntax permits the development of neighborhood-type filters
within a single 3D grid or across multiple 3D grids.

### Functions

The functions currently supported are listed in the table below. The
type of the result is indicated in the last column. *F* means that the
functions always results in a floating point value, *I* means that the
function gives an integer result, and *\** indicates that the result is
float if any of the arguments to the function are floating point values
and integer if all arguments are integer.

| Function                            | Description                                       | Type |
|-------------------------------------|---------------------------------------------------|------|
| abs(x)                              | Return absolute value of x                        | *    |
| acos(x)                             | Inverse cosine of x (result is in degrees)        | F    |
| asin(x)                             | Inverse sine of x (result is in degrees)          | F    |
| atan(x)                             | Inverse tangent of x (result is in degrees)       | F    |
| atan(x, y)                          | Inverse tangent of y/x (result is in degrees)     | F    |
| ceil(x)                             | Smallest integral value not less than x           | *    |
| cos(x)                              | Cosine of x (x is in degrees)                     | F    |
| double(x)                           | Convert x to double-precision floating point      | F    |
| eval([x, y, ...,] z)                | Evaluate values of listed expr, pass results to z |      |
| exp(x)                              | Exponential function of x                         | F    |
| exp(x, y)                           | x to the power y                                  | F    |
| float(x)                            | Convert x to single-precision floating point      | F    |
| floor(x)                            | Largest integral value not greater than x         | *    |
| graph(x, x1, y1[, x2, y2...])       | Convert x to y based on points in a graph         | F    |
| graph2(x, x1[, x2...], y1[, y2...]) | Alternative form of graph()                       | F    |
| if                                  | Decision options:                                 | *    |
| if(x)                               | 1 if x not zero, 0 otherwise                      |      |
| if(x, a)                            | a if x not zero, 0 otherwise                      |      |
| if(x, a, b)                         | a if x not zero, b otherwise                      |      |
| if(x, a, b, c)                      | a if x > 0, b if x is zero, c if x < 0            |      |
| int(x)                              | Convert x to integer [truncates]                  | I    |
| isnull(x)                           | Check if x = NULL                                 |      |
| log(x)                              | Natural log of x                                  | F    |
| log(x, b)                           | Log of x base b                                   | F    |
| max(x, y[, z...])                   | Largest value of those listed                     | *    |
| median(x, y[, z...])                | Median value of those listed                      | *    |
| min(x, y[, z...])                   | Smallest value of those listed                    | *    |
| mod(x, y)                           | Modulus (remainder) of x/y                        | *    |
| mode(x, y[, z...])                  | Mode value of those listed                        | *    |
| nmax(x, y[, z...])                  | Largest value, excluding NULLs                    | *    |
| nmedian(x, y[, z...])               | Median value, excluding NULLs                     | *    |
| nmin(x, y[, z...])                  | Smallest value, excluding NULLs                   | *    |
| nmode(x, y[, z...])                 | Mode value, excluding NULLs                       | *    |
| not(x)                              | 1 if x is zero, 0 otherwise                       |      |
| pow(x, y)                           | x to the power y                                  | *    |
| rand(a, b)                          | Random value x: a <= x < b                        | *    |
| round(x)                            | Round x to nearest integer                        | I    |
| round(x, y)                         | Round x to nearest multiple of y                  |      |
| round(x, y, z)                      | Round x to nearest y*i+z for some integer i       |      |
| sin(x)                              | Sine of x (x is in degrees)                       | F    |
| sqrt(x)                             | Square root of x                                  | F    |
| tan(x)                              | Tangent of x (x is in degrees)                    | F    |
| xor(x, y)                           | Exclusive-or (XOR) of x and y                     | I    |

### Internal variables

| Variable  | Description                                  | Type |
| --------- | -------------------------------------------- | ---- |
| row()     | Current row of moving window                 | I    |
| col()     | Current column of moving window              | I    |
| depth()   | Current depth                                | I    |
| nrows()   | Number of rows in computation region         | I    |
| ncols()   | Number of columns in computation region      | I    |
| ndepths() | Number of depth levels in computation region | I    |
| x()       | Current x-coordinate of moving window        | F    |
| y()       | Current y-coordinate of moving window        | F    |
| z()       | Current z value                              | F    |
| ewres()   | Current east-west resolution                 | F    |
| nsres()   | Current north-south resolution               | F    |
| tbres()   | Current top-bottom resolution                | F    |
| area()    | Area of current cell in square meters        | F    |
| null()    | NULL value                                   |      |

Note, that the row(), col() and depth() indexing starts with 1.

### Floating point values in the expression

Floating point numbers are allowed in the expression. A floating point
number is a number which contains a decimal point:

```sh
    2.3   12.0   12.   .81
```

Floating point values in the expression are handled in a special way.
With arithmetic and logical operators, if either operand is float, the
other is converted to float and the result of the operation is float.
This means, in particular that division of integers results in a
(truncated) integer, while division of floats results in an accurate
floating point value. With functions of type \* (see table above), the
result is float if any argument is float, integer otherwise.

Note: If you calculate with integer numbers, the resulting map will be
integer. If you want to get a float result, add the decimal point to
integer number(s).

If you want floating point division, at least one of the arguments has
to be a floating point value. Multiplying one of them by 1.0 will
produce a floating-point result, as will using float():

```sh
      r3.mapcalc "ratio = float(soil.4 - soil.3) / soil.3)"
```

### NULL support

- Division by zero should result in NULL.

- Modulus by zero should result in NULL.

- NULL-values in any arithmetic or logical operation should result in
  NULL. (however, &&& and \|\|\| are treated specially, as described
  below).

- The &&& and \|\|\| operators observe the following axioms even when x
  is NULL:

  ```sh
      x &&& false == false
      false &&& x == false
      x ||| true == true
      true ||| x == true
  ```

- NULL-values in function arguments should result in NULL (however,
  if(), eval() and isnull() are treated specially, as described below).

- The eval() function always returns its last argument

- The situation for if() is:

  ```sh
  if(x)
      NULL if x is NULL; 0 if x is zero; 1 otherwise
  if(x,a)
      NULL if x is NULL; a if x is non-zero; 0 otherwise
  if(x,a,b)
      NULL if x is NULL; a if x is non-zero; b otherwise
  if(x,n,z,p)
      NULL if x is NULL; n if x is negative;
  z if x is zero; p if x is positive
  ```

- The (new) function isnull(x) returns: 1 if x is NULL; 0 otherwise. The
  (new) function null() (which has no arguments) returns an integer
  NULL.

- Non-NULL, but invalid, arguments to functions should result in NULL.

  ```sh
  Examples:
  log(-2)
  sqrt(-2)
  pow(a,b) where a is negative and b is not an integer
  ```

NULL support: Please note that any math performed with NULL cells always
results in a NULL value for these cells. If you want to replace a NULL
cell on-the-fly, use the isnull() test function in a if-statement.

Example: The users wants the NULL-valued cells to be treated like zeros.
To add maps A and B (where B contains NULLs) to get a map C the user can
use a construction like:

```sh
C = A + if(isnull(B),0,B)
```

**NULL and conditions:**

For the one argument form:

```sh
if(x) = NULL        if x is NULL
if(x) = 0        if x = 0
if(x) = 1        otherwise (i.e. x is neither NULL nor 0).
```

For the two argument form:

```sh
if(x,a) = NULL        if x is NULL
if(x,a) = 0        if x = 0
if(x,a) = a        otherwise (i.e. x is neither NULL nor 0).
```

For the three argument form:

```sh
if(x,a,b) = NULL    if x is NULL
if(x,a,b) = b        if x = 0
if(x,a,b) = a        otherwise (i.e. x is neither NULL nor 0).
```

For the four argument form:

```sh
if(x,a,b,c) = NULL    if x is NULL
if(x,a,b,c) = a        if x > 0
if(x,a,b,c) = b        if x = 0
if(x,a,b,c) = c        if x < 0
```

More generally, all operators and most functions return NULL if \*any\*
of their arguments are NULL.  
The functions if(), isnull() and eval() are exceptions.  
The function isnull() returns 1 if its argument is NULL and 0 otherwise.
If the user wants the opposite, the ! operator, e.g. "!isnull(x)" must
be used.

All forms of if() return NULL if the first argument is NULL. The 2, 3
and 4 argument forms of if() return NULL if the "selected" argument is
NULL, e.g.:

```sh
if(0,a,b) = b    regardless of whether a is NULL
if(1,a,b) = a    regardless of whether b is NULL
```

eval() always returns its last argument, so it only returns NULL if the
last argument is NULL.

**Note**: The user cannot test for NULL using the == operator, as that
returns NULL if either or both arguments are NULL, i.e. if x and y are
both NULL, then "x == y" and "x != y" are both NULL rather than 1 and 0
respectively.  
The behaviour makes sense if the user considers NULL as representing an
unknown quantity. E.g. if x and y are both unknown, then the values of
"x == y" and "x != y" are also unknown; if they both have unknown
values, the user doesn't know whether or not they both have the same
value.

## NOTES

### Usage from command line

Extra care must be taken if the expression is given on the command line.
Some characters have special meaning to the UNIX shell. These include,
among others:

```sh
* ( ) > & |
```

It is advisable to put single quotes around the expression; e.g.:

```sh
'result = volume * 2'
```

Without the quotes, the `*`, which has special meaning to the UNIX
shell, would be altered and *r3.mapcalc* would see something other than
the `*`.

### Multiple computations

In general, it's preferable to do as much as possible in each r3.mapcalc
command using multi-line input.

### Backwards compatibility

For the backwards compatibility with GRASS 6, if no options are given,
it manufactures `file=-` (which reads from stdin), so you can continue
to use e.g.:

```sh
r3.mapcalc < file
```

or:

```sh
r3.mapcalc <<EOF
foo = 1
EOF
```

But unless you need compatibility with previous GRASS versions, use
`file=` explicitly, as stated above.

When the map name contains uppercase letter(s) or a dot which are not
allowed to be in module option names, the *r3.mapcalc* command will be
valid also without quotes:

```sh
r3.mapcalc volume_A=1
r3.mapcalc volume.1=1
```

However, this syntax is not recommended as quotes as stated above more
safe. Using quotes is both backwards compatible and valid in future.

### Interactive input in command line

For formulas that the user enters from standard input (rather than from
the command line), a line continuation feature now exists. If the user
adds a backslash to the end of an input line, *r3.mapcalc* assumes that
the formula being entered by the user continues on to the next input
line. There is no limit to the possible number of input lines or to the
length of a formula.

If the *r3.mapcalc* formula entered by the user is very long, the map
title will contain only some of it, but most (if not all) of the formula
will be placed into the history file for the *result* map.

When the user enters input to *r3.mapcalc* non-interactively on the
command line, the program will not warn the user not to overwrite
existing map layers. Users should therefore take care to assign program
outputs raster map names that do not yet exist in their current mapsets.

### 3D GRID mask handling

*r3.mapcalc* follows the common GRASS behavior of raster mask handling,
so the mask is only applied when reading an existing GRASS raster map.
This implies that, for example, the command:

```sh
r3.mapcalc "volume_amplified = volume * 3"
```

create a map with NULL cells for the masked-out cells if raster mask is
active.

However, when creating a map which is not based on any map, e.g. a map
from a constant:

```sh
r3.mapcalc "volume_const = 200.0"
```

the created raster map is limited only by a computation region but it is
not affected by an active raster mask. This is expected because, as
mentioned above, the mask is only applied when reading, not when writing
a raster map. If the raster mask should be applied in this case, an
`if()` function including the mask raster should be used, e.g. (assuming
the mask is called `MASK`):

```sh
r3.mapcalc "volume_const = if(MASK, 200.0, null())"
```

When testing expressions related to mask handling keep in mind that when
the raster mask is active, you don't see data in masked areas even if
they are not NULL. See *[r3.mask](r3.mask.md)* for details.

### Random number generator initialization

The pseudo-random number generator used by the rand() function can be
initialised to a specific value using the **seed** parameter. This can be
used to replicate a previous calculation.

If rand() function is used and no seed is given,
*r3.mapcalc* will generate a seed, resulting in a different result for
each run.

In either case, the seed will be written to the map's history, and can
be seen using *r.info*.

If you want other people to be able to verify your results, it's
preferable to use the **seed** parameter.

## EXAMPLES

To compute the average of two 3D grids *a* and *b*:

```sh
ave = (a + b)/2
```

To form a weighted average:

```sh
ave = (5*a + 3*b)/8.0
```

To produce a binary representation of the 3D grid *a* so that category 0
remains 0 and all other categories become 1:

```sh
mask = a != 0
```

This could also be accomplished by:

```sh
mask = if(a)
```

To mask 3D grid *b* by 3D grid *a*:

```sh
result = if(a,b)
```

To change all values below 5 to NULL, keep value otherwise:

```sh
newmap = if(map < 5, null(), map)
```

The graph() function allows users to specify a x-y conversion using
pairs of x,y coordinates. In some situations a transformation from one
value to another is not easily established mathematically, but can be
represented by a 2-D graph and then linearly interpolated. The graph()
function provides the opportunity to accomplish this. An x-axis value is
provided to the graph function along with the associated graph
represented by a series of x,y pairs. The x values must be monotonically
increasing (each larger than or equal to the previous). The graph
function linearly interpolates between pairs. Any x value lower the
lowest x value (i.e. first) will have the associated y value returned.
Any x value higher than the last will similarly have the associated y
value returned. Consider the request:

```sh
newmap = graph(map, 1,10, 2,25, 3,50)
```

X (map) values supplied and y (newmap) values returned:

```sh
0, 10
1, 10
1.5, 17.5
2.9, 47.5
4, 50
100, 50
```

## KNOWN ISSUES

The *result* variable on the left hand side of the equation should not
appear in the *expression* on the right hand side.

```sh
mymap = if( mymap > 0, mymap, 0)
```

Any maps generated by a *r3.mapcalc* command only exist after the entire
command has completed. All maps are generated concurrently, row-by-row
(i.e. there is an implicit "for row in rows {...}" around the entire
expression). Thus the `#`, `@`, and `[ ]` operators cannot be used on a
map generated within same *r3.mapcalc* command run.

```sh
newmap = oldmap * 3.14
othermap = newmap[-1, 0] / newmap[1, 0]
```

Continuation lines must end with a `\` and have *no* trailing white
space (blanks or tabs). If the user does leave white space at the end of
continuation lines, the error messages produced by *r3.mapcalc* will be
meaningless and the equation will not work as the user intended. This is
particularly important for the `eval()` function.

Currently, there is no comment mechanism in *r3.mapcalc*. Perhaps adding
a capability that would cause the entire line to be ignored when the
user inserted a \# at the start of a line as if it were not present,
would do the trick.

The function should require the user to type "end" or "exit" instead of
simply a blank line. This would make separation of multiple scripts
separable by white space.

*r3.mapcalc* does not print a warning in case of operations on NULL
cells. It is left to the user to utilize the `isnull()` function.

## REFERENCES

**[r.mapcalc: An Algebra for GIS and Image
Processing](https://grass.osgeo.org/gdp/raster/mapcalc-algebra.pdf)**,
by Michael Shapiro and Jim Westervelt, U.S. Army Construction
Engineering Research Laboratory (March/1991).

**[Performing Map Calculations on GRASS Data: r.mapcalc Program
Tutorial](https://grass.osgeo.org/history_docs/mapcalc.pdf)**, by Marji
Larson, Michael Shapiro and Scott Tweddale, U.S. Army Construction
Engineering Research Laboratory (December 1991)

Neteler, M. (2001): Volume modelling of soils using GRASS GIS 3D
tools. - in: Brovelli, M. (ed.)(2001): The Geomatics Workbook N. 2.
Politecnico di Milano, Italy (ISSN 1591-092X)
([PDF](http://geomatica.como.polimi.it/workbooks/n2/articoli/mn2.pdf))

## SEE ALSO

*[g.region](g.region.md), [r3.colors](r3.colors.md),
[r.mapcalc](r.mapcalc.md)*

## AUTHORS

Tomas Paudits & Jaro Hofierka, funded by GeoModel s.r.o., Slovakia  
<tpaudits@mailbox.sk>,
[hofierka@geomodel.sk](MAILTO:hofierka@geomodel.sk)

Glynn Clements
