## DESCRIPTION

This numerical module calculates implicit transient and steady state,
confined groundwater flow in three dimensions based on volume maps and
the current 3D region settings. All initial- and boundary-conditions
must be provided as volume maps. The unit of the current coordinate
reference system must be meters.

This module is sensitive to mask settings. All cells which are outside
the mask are ignored and handled as no flow boundaries.

The module calculates the piezometric head and optionally the water
balance for each cell and the groundwater velocity field in 3
dimensions. The vector components can be visualized with ParaView if
they are exported with *r3.out.vtk*.

The groundwater flow will always be calculated transient. For steady
state computation the user should set the timestep to a large number
(billions of seconds) or set the specific yield raster map to zero.

## NOTES

The groundwater flow calculation is based on Darcy's law and a numerical
implicit finite volume discretization. The discretization results in a
symmetric and positive definite linear equation system in form of *Ax =
b*, which must be solved. The groundwater flow partial differential
equation is of the following form:

(dh/dt)\*S = div (K grad h) + q

In detail for 3 dimensions:

(dh/dt)\*S = Kxx \* (d^2h/dx^2) + Kyy \* (d^2h/dy^2) + Kzz \*
(d^2h/dz^2) + q

- h -- the piezometric head im meters \[m\]
- dt -- the time step for transient calculation in seconds \[s\]
- S -- the specific yield \[1/m\]
- b -- the bottom surface of the aquifer meters \[m\]
- Kxx -- the hydraulic conductivity tensor part in x direction in meter
  per second \[m/s\]
- Kyy -- the hydraulic conductivity tensor part in y direction in meter
  per seconds \[m/s\]
- Kzz -- the hydraulic conductivity tensor part in z direction in meter
  per seconds \[m/s\]
- q - inner source/sinc in \[1/s\]

Two different boundary conditions are implemented, the Dirichlet and
Neumann conditions. By default the calculation area is surrounded by
homogeneous Neumann boundary conditions. The calculation and boundary
status of single cells can be set with the status map, the following
cell states are supported:

- 0 == inactive - the cell with status 0 will not be calculated, active
  cells will have a no flow boundary to an inactive cell
- 1 == active - this cell is used for groundwater calculation, inner
  sources can be defined for those cells
- 2 == Dirichlet - cells of this type will have a fixed piezometric head
  value which do not change over time

Note that all required raster maps are read into main memory.
Additionally the linear equation system will be allocated, so the memory
consumption of this module rapidely grow with the size of the input
maps.

The resulting linear equation system *Ax = b* can be solved with several
solvers. An iterative solvers with sparse and quadratic matrices support
is implemented. The conjugate gradients method with (pcg) and without
(cg) precondition. Additionally a direct Cholesky solver is available.
This direct solver only work with normal quadratic matrices, so be
careful using them with large maps (maps of size 10.000 cells will need
more than one Gigabyte of RAM). The user should always prefer to use a
sparse matrix solver.

## EXAMPLE 1

This small script creates a working groundwater flow area and data. It
cannot be run in a lat/lon project.

```sh
# set the region accordingly
g.region res=25 res3=25 t=100 b=0 n=1000 s=0 w=0 e=1000 -p3

#now create the input raster maps for a confined aquifer
r3.mapcalc expression="phead = if(row() == 1 && depth() == 4, 50, 40)"
r3.mapcalc expression="status = if(row() == 1 && depth() == 4, 2, 1)"
r3.mapcalc expression="well = if(row() == 20 && col() == 20 && depth() == 2, -0.25, 0)"
r3.mapcalc expression="hydcond = 0.00025"
r3.mapcalc expression="syield = 0.0001"
r.mapcalc  expression="recharge = 0.0"

r3.gwflow solver=cg phead=phead statuyield=status hc_x=hydcond hc_y=hydcond  \
   hc_z=hydcond sink=well yield=syield r=recharge output=gwresult dt=8640000 vx=vx vy=vy vz=vz budget=budget

# The data can be visualized with ParaView when exported with r3.out.vtk
r3.out.vtk -p in=gwresult,status,budget vector=vx,vy,vz out=/tmp/gwdata3d.vtk

#now load the data into ParaView
paraview --data=/tmp/gwdata3d.vtk
```

## EXAMPLE 2

This will create a nice 3D model with geological layer with different
hydraulic conductivities. Make sure you are not in a lat/lon projection.

```sh
# set the region accordingly
g.region res=15 res3=15 t=500 b=0 n=1000 s=0 w=0 e=1000

#now create the input raster maps for a confined aquifer
r3.mapcalc expression="phead = if(col() == 1 && depth() == 33, 50, 40)"
r3.mapcalc expression="status = if(col() == 1 && depth() == 33, 2, 1)"
r3.mapcalc expression="well = if(row() == 20 && col() == 20 && depth() == 3, -0.25, 0)"
r3.mapcalc expression="well = if(row() == 50 && col() == 50 && depth() == 3, -0.25, well)"
r3.mapcalc expression="hydcond = 0.0025"
r3.mapcalc expression="hydcond = if(depth() < 30 && depth() > 23 && col() < 60, 0.000025, hydcond)"
r3.mapcalc expression="hydcond = if(depth() < 20 && depth() > 13 && col() >  7, 0.000025, hydcond)"
r3.mapcalc expression="hydcond = if(depth() < 10 && depth() >  7 && col() < 60, 0.000025, hydcond)"
r3.mapcalc expression="syield = 0.0001"

r3.gwflow solver=cg phead=phead statuyield=status hc_x=hydcond hc_y=hydcond  \
   hc_z=hydcond sink=well yield=syield output=gwresult dt=8640000 vx=vx vy=vy vz=vz budget=budget

# The data can be visualized with paraview when exported with r3.out.vtk
r3.out.vtk -p in=gwresult,status,budget,hydcond,well vector=vx,vy,vz out=/tmp/gwdata3d.vtk

#now load the data into paraview
paraview --data=/tmp/gwdata3d.vtk
```

## SEE ALSO

*[r.gwflow](r.gwflow.md), [r.solute.transport](r.solute.transport.md),
[r3.out.vtk](r3.out.vtk.md)*

## AUTHOR

Sören Gebbert

This work is based on the [Diploma Thesis of Sören Gebbert](https://grass.osgeo.org/gdp/hydrology/gebbert2007_diplom_stroemung_grass_gis.pdf)
at Technical University Berlin, Germany.
