## DESCRIPTION

The module *r3.out.vtk* outputs 3D raster maps into VTK-ASCII format.
Maps are valid 3D raster maps in the current mapset search path. The
*output* parameter is the name of a VTK-ASCII file which will be written
in the current working directory. If *output* is not specified then
standard output (stdout) is used. The module is sensitive to region
settings (set with *g.region*).

## NOTES

This module generates *structured points* with *celldata* (default) or
*pointdata*. If top and bottom surfaces are requested an *unstructured
grid* with *celldata* or a *structured grid* with *pointdata* is
generated. This data is put in a simple VTK-ASCII file. Neither XML nor
binary output are supported. It is possible to choose more then one 3D
raster map to be written in the VTK-ASCII file. Each *celldata* is named
as the 3D raster map it represents. The user can visualize this file
with the *[VTK Toolkit](https://vtk.org/)*,
*[ParaView](https://www.paraview.org/)* and
*[MayaVi](https://github.com/enthought/mayavi)* which are based on VTK.
In case of 3D raster map with partially no data, the threshold filter in
ParaView can be used to visualize the valid data. Just filter all data
which is greater/lesser than the chosen null value in the VTK-ASCII
file.

The top and bottom region values are expected in meters. If a
Latitude-Longitude (LL) coordinates are used, the elevation value for
each voxel will be converted into degrees.

The *input*, *rgbmaps* and *vectormaps* parameters are optional, so only
the geometry can be exported.

If the user defines *top* and *bottom* and the 2D and 3D region values
differ, the 2D resolution will be adjusted to the 3D resolution. The
elevation maps are expected in meters. If Lat/Long coordinates are used,
the elevation will automatically converted into degree. If the surface
and bottom maps are in a different unit than meters, use the scale
parameter to convert them into meters.

The RGB voxel data can be created from 2D raster maps (Landsat TM
images) with *[r.to.rast3](r.to.rast3.md)*. The values of the RGB maps
must be within 0 and 255. If not, the values are automatically set to 0
and warnings will be printed to stderr.

The vector data is created from three 3D raster maps. Each map
represents a vector component. So x, y and z components are required in
this order. This data can be visualized with Glyph3d or StreamTracer
filters within Paraview.

If the *-c* flag is used and the data should be visualised together with
other data exported via *\*.out.vtk* modules, be sure the *-c* flag was
also set in these modules. But this will only work with data from the
SAME location (the reference point for the coordinates transformation is
based on the center point of the default region).

### Difference between point- and celldata

*r3.out.vtk* can export 3D raster maps with different representations.

- *pointdata* -- the cells/values are represented by the center of the
  cell. Instead of cells, points are created. Each point can hold
  different values, but the user can only visualize one value at a time.
- *celldata* The cells are created with the same height, width and depth
  as in GRASS. Each cell can hold different values, but the user can
  only visualize one value at a time.

## EXAMPLES

### Simple Spearfish example

```sh
g.region -d
g.region res=150 res3=150 t=80 b=0 tbres=10
r.mapcalc "bottom = 1800. - elevation.10m"

# synthetic data, could be geological structures:
r3.mapcalc "map3d = row()+col()+depth()"

#export of volume to VTK:
r3.out.vtk -s input=map3d top=elevation.10m bottom=bottom output=/tmp/out.vtk

# visualize in paraview or other VTK viewer:
paraview --data=/tmp/out.vtk
```

### Spearfish example with RGB data

```sh
#set the region
g.region -d
g.region n=4926970 s=4914857 w=591583 e=607793 res=50 res3=50 t=80 b=0 tbres=10

#create a bottom surface
r.mapcalc "bottom = 1800. - elevation.10m"

# synthetic data, could be geological structures:
r3.mapcalc "map3d = row()+col()+depth()"

#get some satellite images with r.in.onearth
r.in.onearth -l output=Sat tmband=Red
r.in.onearth -l output=Sat tmband=IR1
r.in.onearth -l output=Sat tmband=IR2

#Convert the 2D maps to 3D raster maps with r.to.rast3
r.to.rast3 input=SatLandsatTM_Red output=SatLandsatTM_Red
r.to.rast3 input=SatLandsatTM_IR1 output=SatLandsatTM_IR1
r.to.rast3 input=SatLandsatTM_IR2 output=SatLandsatTM_IR2

#export of volume to VTK:
r3.out.vtk -s rgbmaps=SatLandsatTM_IR1,SatLandsatTM_IR2,SatLandsatTM_Red
input=map3d top=elevation.10m bottom=bottom output=/tmp/out.vtk

# visualize in paraview or other VTK viewer:
paraview --data=/tmp/out.vtk
```

### Spearfish example with vector data

```sh
# set the region
g.region -d
g.region n=4926970 s=4914857 w=591583 e=607793 res=50 res3=50 t=80 b=0 tbres=10

# create a bottom surface
r.mapcalc "bottom = 1800. - elevation.10m"

# synthetic data, could be geological structures:
r3.mapcalc "map3d = row()+col()+depth()"

# synthetic vector data, could be groundwater stream vectors
r3.mapcalc "x_part = sin(row())"
r3.mapcalc "y_part = cos(col())"
r3.mapcalc "z_part = sin(depth())"


# export the stuff data to VTK:
r3.out.vtk -s vectormaps=x_part,y_part,z_part input=map3d top=elevation.10m
bottom=bottom output=/tmp/out.vtk

# visualize in paraview or other VTK viewer:
paraview --data=/tmp/out.vtk
# Now use the Glyph and Stream-Trace Filter to get nice vectors and streamlines
```

### Slovakia3d example

```sh
#reduce resolution:
g.region -dp3 res=1000 res3=1000
r.mapcalc "bottom = 100"

#export of volume to VTK:
r3.out.vtk -s in=precip3d.500z50 top=dem500 bottom=bottom \
   output=/path/to/slovakia3d.vtk

# visualize in paraview or other VTK viewer:
paraview --data=/path/to/slovakia3d.vtk
# set Display style to 'surface#
# set Actor Control z to 10
```

## SEE ALSO

*[r.out.vtk](r.out.vtk.md), [r3.out.ascii](r3.out.ascii.md),
[g.region](g.region.md)*

## AUTHOR

Sören Gebbert
